(ns hub.user.service
  "Service implementation details."
  (:require [clj-time.coerce :as tc]
            [clj-time.core :refer [now]]
            [clojure.string :as str]
            [clojurewerkz.scrypt.core :as sc]
            [com.stuartsierra.component :as c]
            [hub.user.rethink :as ur :refer [connect]]
            [hub.user.schema :as us]
            [rethinkdb.query :as r]
            [schema.core :as s])
  (:import [com.stuartsierra.component Lifecycle]))

;; ## Helpers

(defn user-table
  "Returns a command for the user table, suitable for chaining with other commands."
  []
  (r/table (r/db (ur/db-name)) "user"))

(defmacro run-user
  "Runs a command on the user table."
  [command]
  `(ur/run (-> (user-table)
               ~command)))

(s/def UnixTime
  (s/named s/Int "Instant in time defined as the number of ms
  since midnight UTC Jan 1 1970."))

(s/defn unix-time :- UnixTime
  "Returns the current UnixTime"
  []
  (tc/to-long (now)))

;; ## Setup

(s/defn setup!
  "Basic setup for the user table.

  TODO: Should we index on email as well? Pending Users won't have
  emails OR usernames."
  [mode :- (s/enum :test :live)]
  (let [db-name (ur/db-name)
        idx-exists? (fn [idx-name]
                      (-> (r/index-list
                           (user-table))
                          (r/contains idx-name)))
        create-downcased-idx (fn [idx-name kw]
                               (r/index-create
                                (user-table)
                                idx-name
                                (r/fn [row]
                                  (r/downcase
                                   (r/get-field row kw)))))]
    ;;create racehub database, if needed
    (ur/run (ur/try-create db-name))

    ;;create user table, if needed
    (ur/run (ur/try-table db-name "user"))

    ;;create username index, if needed
    (ur/run (r/branch (idx-exists? "username")
                      true
                      (create-downcased-idx "username" :username)))))

;; ## GET API

(s/defn get-user-by-id :- (s/maybe us/User)
  [id :- us/UserID]
  (run-user (r/get id)))

(s/defn get-user-by-email :- (s/maybe us/User)
  [email :- s/Str]
  (first (run-user (r/filter {:email email}))))

(s/defn get-many-users-map :- {us/UserName us/User}
  "Takes in a collection of usernames, and returns a map of username
  to user doc for all of them."
  [usernames :- [us/UserName]]
  (into {} (for [d (run-user (r/get-all (set usernames) {:index "username"}))]
             [(:username d) d])))

;; ## CREATE Helpers

(s/defn username-valid? :- s/Bool
  [s :- s/Str]
  (boolean
   (re-matches #"[a-zA-Z0-9]{3,20}" s)))

(s/defschema GenOptions
  "Options for username generation."
  {:email s/Str
   (s/optional-key :first-name) s/Str
   (s/optional-key :last-name) s/Str})

(s/defn random-usernames
  "Don't put an output schema on this! Returns an infinite sequence
    of usernames created by sticking the prefix onto the beginning of
    random numbers below the supplied maximum value."
  [prefix :- s/Str
   max :- s/Int]
  (map (partial str prefix) (repeatedly #(rand-int max))))

(s/defn winner :- (s/maybe s/Str)
  "Queries the database for all supplied username
    candidates. Returns the first candidate that doesn't already exist
    in the database, or nil if they all do."
  [candidates :- [(s/maybe s/Str)]]
  (let [candidates (->> candidates
                        (remove (some-fn empty? (complement username-valid?)))
                        (map str/lower-case))
        fetched (get-many-users-map candidates)]
    (some (fn [k] (when-not (fetched k) k)) candidates)))

(s/defn generate-username :- s/Str
  "Generates a random username for the supplied GenOptions map of
    seeding options. Tries concatenating the first and last names,
    taking just the first initial and last name, and the email
    prefix (before the @ sign). After that, attempts to generate a
    username.  NOT guaranteed to terminate, but extremely likely :)"
  [{:keys [email first-name last-name]} :- GenOptions]
  (let [email-prefix            (first (str/split email #"@"))
        first-initial-last-name (str (first first-name) last-name)
        first-tries [email-prefix
                     first-initial-last-name
                     (str first-name last-name)]
        un-prefix (or email-prefix last-name)]
    (loop [candidates first-tries seed 1000]
      (or (winner candidates)
          (recur (take 10 (random-usernames un-prefix seed))
                 (* seed 10))))))

(defmulti generate-user
  "Generates a user document off of the information provided for the
  supplied type. Signup gives username and password only, for
  example. Facebook gives a bit more."
  (fn [type data] type))

(defn encrypt
  "Encrypts a string value using scrypt.
   Arguments are:
   raw (string): a string to encrypt
   :n (integer): CPU cost parameter (default is 16384)
   :r (integer): RAM cost parameter (default is 8)
   :p (integer): parallelism parameter (default is 1)
   The output of SCryptUtil.scrypt is a string in the modified MCF format:
   $s0$params$salt$key
   s0     - version 0 of the format with 128-bit salt and 256-bit derived key
   params - 32-bit hex integer containing log2(N) (16 bits), r (8 bits), and p (8 bits)
   salt   - base64-encoded salt
   key    - base64-encoded derived key"
  [raw & {:keys [n r p]
          :or {n 16384 r 8 p 1}}]
  (sc/encrypt raw n r p))

(s/defmethod generate-user :signup
  [_ {:keys [email password]} :- {:email s/Str, :password s/Str}]
  {:password (encrypt password)
   :username (generate-username {:email email})
   :email {:address email, :verified? false}
   :profile {}})

(s/defmethod generate-user :default
  [_ m :- {s/Any s/Any}]
  (cond-> m
    (:username m) (update :username str/lower-case)
    (-> m :email :address) (update-in [:email :address] str/lower-case)
    (:password m) (update :password encrypt)))

(s/defschema SignupType
  (s/enum :signup :facebook :default))


;; ## MUTATING API

(s/defn create! :- us/User
  "Creates a user document using the supplied method; returns the
  created user record."
  ([data] (create! :default data))
  ([type :- SignupType data]
   (let [user (generate-user type data)
         db-fields {:created-at (unix-time)}
         create-result (run-user (r/insert (merge user db-fields)))]
     (merge user
            {:id (first (:generated_keys create-result))}
            db-fields))))

(s/defn update-by-id! :- us/User
  "Updates the user with the given id by merging it with the new map. Cant update id."
  [user-id :- us/UserID
   updates :- {s/Any s/Any}]
  (let [db-fields {:updated-at (unix-time)}
        updated (run-user (-> (r/get user-id)
                         (r/update (r/fn [user-doc]
                                     (merge (dissoc user-doc :temp-var)
                                            updates
                                            db-fields)))))]
    (get-user-by-id user-id)))

(s/defn update! :- us/User
  "Updates the given user-doc by merging it with the new map. Cant update id."
  [user-doc :- us/User
   updates :- {s/Any s/Any}]
  (update-by-id! (:id user-doc) updates))

(s/defn put! :- us/User
  "Puts the user-doc with the desired changes. Cant change id."
  [{:keys [id] :as new-user} :- us/User]
  (let [updated (merge new-user {:updated-at (unix-time)})]
    (run-user (-> (r/get id)
                  (r/replace updated)))
    updated))

(s/defn delete-by-id! :- s/Bool
  "Deletes the given user, returns true if successful."
  [user-id :- us/UserID]
  (if-let [{:keys [deleted]} (run-user (-> (r/get user-id)
                                      r/delete))]
    (pos? deleted)
    false))

; ## User Info Queries
  (def super-admins
    #{"sritchie" "dpetrovics" "thornsby"})

  (s/defn super-admin? :- s/Bool
    [user-doc :- (s/maybe us/User)]
    (contains? super-admins (:username user-doc)))


;; ## Service Startup

(s/defn rethinkdb :- Lifecycle
  "Lifecycle component for the user service database."
  ([mode] (rethinkdb mode "racehub" {}))
  ([mode :- (s/enum :test :live)
    db-name :- s/Str
    spec :- ur/RethinkSpec]
   (reify c/Lifecycle
     (start [_] (do (println "Starting RethinkDB...")
                    (reset! ur/conf {:db-name (if (= mode :test)
                                                (str db-name "_test")
                                                db-name)})
                    (setup! mode)))
     (stop [_] (do (println "Stopping RethinkDB...")
                   (when (= mode :test)
                     (ur/run (r/db-drop (str db-name "_test"))))
                   (reset! ur/conf nil))))))
