(ns hub.user.client
  "User Service client."
  (:require [hub.user.schema :as u]
            [schema.core :as s]))

(s/defschema Err
  {:error
   {:type s/Str
    (s/optional-key :param) s/Str
    (s/optional-key :message) s/Str}})

(def ID s/Str)

(s/defschema User
  "Something like this. Doesn't quite have to match the DB
  representation."
  {:id ID
   :username s/Str
   :email u/EMailMap
   :password-set? (s/named s/Bool "Is the user's PW set?")
   :roles #{(s/enum "user" "superadmin")}
   :oauth (s/named {s/Keyword s/Str}
                   "Map of oauth provider to the name given by that
                   provider.")
   ;; profile? Photos?
   })

;; # User Service
;;
;; The facebook API stuff needs to move over here in some config
;; namespace.

;; ## Create

(s/defn create! :- (s/either User Err)
  "Generates a user off of the supplied username and email."
  [m :- {:email s/Str :password s/Str}])

(s/defn create-pending! :- (s/either User Err)
  "Generates a pending user instance. Returns an error if a user
  exists with the supplied email."
  [m :- {:name s/Str
         (s/optional-key :email) s/Str
         s/Any s/Any}])

;; ## Facebook
;;
;; These are pretty easy to extend to the other oauth providers.

(s/defn register-facebook! :- (s/either User Err)
  "Generates a user off of the supplied facebook
   information. Optionally takes a UserID to associate the FaceBook
   info with.

  - If a user already exists with that facebook id, upgrades their
    token.

  - errors if another user is using the facebook account's email and
    ALREADY has another linked facebook account.

  - otherwise, associates this facebook account with that user and
    merges the new facebook info into that user's profile.

  (see signup/get-or-create-via-facebook! for impl details)"
  [m :- {:access-token s/Str
         :facebook-id s/Str
         (s/optional-key :id) ID}])

(s/defn unregister-facebook! :- User
  "Removes Facebook info from the supplied user. Returns that user."
  [id :- ID])

;; ## Get

(s/defschema LookupType
  "Ways to search for some user."
  (s/enum :username :id :email :facebook-email :facebook-id))

(s/defn multiget :- {s/Any User}
  "Returns a map of lookup value -> the user, if found. NOTE that this
  will fuck up if you send in the same value with different
  types. Maybe a problem?"
  [m :- [{:value s/Any, :type LookupType}]])

;; ## Update

(s/defn update-user! :- (s/either User Err)
  "Updates the user. If a new value for password, email or username is
  passed in, updates those fields appropriately (password gets hashed,
  email verification gets sent). Errors if the user doesn't exist."
  [id :- ID m :- {s/Keyword s/Any}])

;; ## Authentication

(s/defn reset-password :- (s/either User Err)
  "Generate a password reset and activation code for the supplied
  user. Errors if the user doesn't exist."
  [id :- ID])

(s/defn password-valid? :- (s/either s/Bool Err)
  "Does the user with the supplied ID's password match this password?
  Errors if the password isn't set."
  [id :- ID password :- s/Str])

;; ## User Merge

(s/defn merge-user :- (s/either User Err)
  "Merges the sub-id user into the primary-id user. Returns the merged
  user. Errors if either user doesn't exist.

  This can be used to merge real users, OR to merge pending users
  together."
  [primary-id :- ID
   sub-id :- ID])

;; ## Missing!
;;
;; - profile updates and gets
