(ns hub.queue.client
  (:require [com.stuartsierra.component :as c]
            [hub.queue.service :as q]
            [schema.core :as s]
            [taoensso.carmine :as car])
  (:import [com.stuartsierra.component Lifecycle]))
"Client for the RaceHub message bus."

(s/defn publish! :- s/Int
  "Publishes the given payload on the topic. Returns the number of subscribers."
  [topic :- s/Str
   payload :- s/Any]
  (if (not-empty topic)
    (try  (q/wcar (car/publish topic payload))
          (catch Exception e
            e))
    (throw (ex-info "Empty topic." {:type "empty_topic"}))))


(s/defn subscribe! :- (s/named (s/=> (s/pred nil?)) "unsubscribe fn")
  "Subscribes to the given topic, and executes the handler on any
  messages that arrive. Returns a function of no args that can be
  called to unsubscribe from this topic."
  [topic :- s/Str
   handler]
  (if (not-empty topic)
    (try (let [listener (car/with-new-pubsub-listener (:spec @q/conf)
                          {topic (fn f1 [[msg-type _ msg]] (when (= msg-type "message")
                                                            (handler msg)))}
                          (car/subscribe  topic))]
           (println "Queue: Subscribed to " topic)
           (fn []
             (println "Queue: Unsubscribing from " topic)
             (car/with-open-listener listener (car/unsubscribe))))
         (catch Exception e
           e))
    (throw (ex-info "Empty topic." {:type "empty_topic"}))))

;; ## Service Startup
(s/defschema RedisSpec
  {(s/optional-key :host) s/Str
   (s/optional-key :port) s/Int
   (s/optional-key :password) s/Str
   (s/optional-key :timeout-ms) s/Int
   (s/optional-key :db) s/Int})

(s/defn queue-client :- Lifecycle
  "Client for the queue service."
  ([] (queue-client {}))
  ([spec :- RedisSpec]
   (reify c/Lifecycle
     (start [_] (reset! q/conf {:pool {} :spec spec}))
     (stop [_] (reset! q/conf nil)))))
