(ns hub.photo.setup
  "Setup of photo service - create database, tables, indices."
  (:require [hub.util.rethink :as ur :refer [args]]
            [hub.photo.service :as ps]
            [rethinkdb.query :as r]
            [schema.core :as s]))

(s/defn setup!
  "Basic setup for the photos table in RethinkDB."
  [db-name :- s/Str]
  (let [maybe-index (fn [table-name index-name query-fn & opts]
                      (apply ur/maybe-create-index
                             index-name
                             (r/table (r/db db-name) table-name)
                             query-fn
                             opts))
        photo-index (partial maybe-index ps/photo-table)
        privacy-index (partial maybe-index ps/privacy-table)
        info-index (partial maybe-index ps/race-info)]

    ;; Try to create the racehub database.
    (ur/run (ur/try-create db-name))

    ;; Create the tables.
    (let [tables [ps/photo-table ps/privacy-table ps/race-info]]
      (doseq [table tables]
        (ur/run (ur/try-table db-name table))))

    (photo-index "by-race-id"
                 (r/fn [row]
                   (r/get-field row :race-id)))

    (photo-index "by-bib-id"
                 (r/fn [row]
                   (r/map (r/get-field row :tags)
                          (r/fn [tag]
                            [(r/get-field row :race-id)
                             (r/get-field tag :bib)])))
                 {:multi true})

    (privacy-index "by-user-id"
                   (r/fn [row] (r/get-field row :user-id)))

    (info-index "by-race-id"
                (r/fn [row] (r/get-field row :race-id)))

    ;; Allows you to get bib mappings by UserID.
    (info-index "by-user-id"
                (let [mapping #(r/get-field % :mapping)]
                  (r/fn [row]
                    (r/map (r/keys (mapping row))
                           (r/fn [k]
                             (-> (mapping row)
                                 (r/get-field k)
                                 (r/get-field :user-id))))))
                {:multi true})))
