(ns hub.photo.service
  "Photo Service implementation details."
  (:require [com.stuartsierra.component :as c]
            [hub.photo.config :as conf]
            [hub.photo.facebook :as fb]
            [hub.photo.schema :as ps]
            [hub.util.rethink :as ur]
            [rethinkdb.query :as r]
            [rethinkdb.net :as n]
            [schema.core :as s])
  (:import [com.stuartsierra.component Lifecycle]
           [rethinkdb.net Cursor]))

(def photo-table "photo")
(def privacy-table "photo_privacy")
(def race-info "race_info")

(defmacro run-photo
  "Run a command on the photo table."
  [& forms]
  `(ur/run
     (-> (r/table photo-table)
         ~@forms)))

(defmacro run
  "Run a command on the configured database."
  [& forms]
  `(ur/run
     (-> (r/db (ur/db-name))
         ~@forms)))

(defn map-values [f m]
  (into {} (for [[k v] m]
             [k (f v)])))

(def multiget->get
  "Takes a function of [id] -> {id item} and returns a function of id
  -> (s/maybe item)."
  (fn [f] (fn [x] (second (first (f [x]))))))

;; ## User Info
(s/defn multiget-user-info :- {ps/UserID ps/UserTagDoc}
  [ids :- [ps/UserID]]
  (let [docs (if-let [ids (not-empty (distinct ids))]
               (run
                 (r/table privacy-table)
                 (r/get-all ids {:index "by-user-id"})
                 (r/coerce-to "ARRAY"))
               [])]
    (into {} (map (fn [m] [(:user-id m) m]) docs))))

(def get-user-info
  "If the user has privacy settings registered, returns those
  settings."
  (multiget->get multiget-user-info))

(s/defn set-user-info! :- ps/UserTagDoc
  "Sets the user info fields."
  [user-id :- ps/UserID
   opts :-  {(s/optional-key :privacy-level) ps/Privacy
             (s/optional-key :facebook-id) s/Str}]
  (if-let [result (first
                   (run
                     (r/table privacy-table)
                     (r/get-all [user-id] {:index "by-user-id"})
                     (r/coerce-to "ARRAY")))]
    (let [doc (merge result opts)]
      (run
        (r/table privacy-table)
        (r/replace (merge result opts)))
      doc)
    (let [[k] (run
                (r/table privacy-table)
                (r/insert (assoc opts :user-id user-id))
                (r/get-field :generated_keys))]
      (run
        (r/table privacy-table)
        (r/get k)))))

;; ## Mappings

(s/defn get-infos :- {s/Str ps/RaceInfo}
  "Multiget for bib mappings. Returns a map of race-id -> BibMapping."
  [race-ids :- [s/Str]]
  (if-let [ids (not-empty (distinct race-ids))]
    (let [mappings (->> (run
                          (r/table race-info)
                          (r/get-all ids {:index "by-race-id"})
                          (r/coerce-to "ARRAY"))
                        (map (fn [doc]
                               (letfn [(string-keys [m]
                                         (into {} (map (fn [[k v]] [(name k) v]) m)))]
                                 (update doc :mapping string-keys)))))]
      (into {} (map (fn [m] [(:race-id m) m]) mappings)))
    {}))

(def get-race-info
  "If the race has info on albums and mappings registered, returns
  those settings."
  (multiget->get get-infos))

(s/defn hydrate-users :- [ps/Photo]
  "Fills in the UserTag field on the tags in each photo."
  [photos :- [ps/Photo]]
  (let [race-ids (map :race-id photos)
        race-id->info (get-infos race-ids)
        tag-user (fn [mapping {:keys [bib] :as tag}]
                   (merge tag (when-let [user-m (get mapping bib)]
                                {:user-tag {:user-id (:user-id user-m)}})))
        user-tags (fn [photo]
                    (if-let [{:keys [mapping]} (race-id->info (:race-id photo))]
                      (let [tag-fn (fn [tags]
                                     (map (partial tag-user mapping) tags))]
                        (update photo :tags tag-fn))
                      photo))
        with-users (map user-tags photos)
        user-ids (mapcat (fn [photo]
                           (->> (:tags photo)
                                (map (comp :user-id :user-tag))
                                (remove nil?)))
                         with-users)
        user-id->info (multiget-user-info user-ids)
        add-privacy (fn [photo {:keys [bib user-tag] :as tag}]
                      (if-not user-tag
                        tag
                        (let [default (-> (:race-id photo)
                                          (race-id->info)
                                          :mapping
                                          (get bib)
                                          :default-privacy)
                              info (user-id->info (:user-id user-tag))]
                          (update tag :user-tag
                                  (fn [tag]
                                    (-> tag
                                        (merge (when-let [id (:facebook-id info)]
                                                 {:facebook-id id}))
                                        (assoc :privacy-level
                                               (or (:privacy-level info)
                                                   default
                                                   (conf/default-privacy)))))))))]
    (map (fn [photo]
           (update photo :tags (partial map (partial add-privacy photo))))
         with-users)))

(s/defn multiget :- {ps/ID ps/Photo}
  "Same as the client's multiget, but handles hydrating the bib
  mappings."
  [ids :- [ps/ID]]
  (if-let [ids (not-empty (distinct ids))]
    (->> (run-photo (ur/get-all ids))
         (hydrate-users)
         (map (juxt :id identity))
         (into {}))
    {}))

;; ## Tagging

(s/defn tag-facebook-photos! :- [{:success s/Bool}]
  "Tags all users in all photos on facebook. Accepts a sequence of
  photos, unhydrated by default. Optionally you can supply `true` as
  the second param to signal that they're already hydrated."
  ([photos :- [ps/Photo]]
   (tag-facebook-photos! photos false))
  ([photos :- [ps/Photo] pre-hydrated? :- s/Bool]
   (let [photos (if pre-hydrated?
                  photos
                  (hydrate-users photos))
         tags (for [photo photos :when (:facebook-id photo)]
                {:photo-fb-id (:facebook-id photo)
                 :user-fb-ids (map (comp :facebook-id :user-tag)
                                   (:tags photo))})]
     (if (conf/prod?)
       (fb/tag-photos-prod tags)
       (for [{:keys [photo-fb-id user-fb-ids]} tags
             id user-fb-ids]
         {:success true})))))
