(ns hub.photo.facebook
  (:require [cheshire.core :refer [parse-string generate-string]]
            [hub.photo.config :as conf]
            [hub.photo.schema :as ps]
            [org.httpkit.client :as http]
            [hub.util.facebook :as fb]
            [schema.core :as s])
  (:import [java.util UUID]))

;; ## Schema

(s/defschema LinkedInput
  (assoc ps/PhotoInput
         :facebook-id ps/FacebookID
         :album-id ps/AlbumID))

;; ## API

(def query-string
  @#'http/query-string)

;; ### Photo Uploading

(s/defn ^:private link-photos-dev :- [LinkedInput]
  [album-id :- ps/AlbumID
   cdn-prefix :- s/Str
   inputs :- [ps/PhotoInput]]
  (map #(assoc %
               :facebook-id (str (UUID/randomUUID))
               :album-id album-id)
       inputs))

(s/defn ^:private link-photos-prod :- [LinkedInput]
  [album-id :- ps/AlbumID
   cdn-prefix :- s/Str
   inputs :- [ps/PhotoInput]]
  (let [{:keys [page-id page-admin-token]} (conf/facebook-config)
        relative-url (fn [opts]
                       (str (str album-id "/photos") "?"
                            (query-string opts)))

        opts {:query-params
              {:access_token page-admin-token
               :batch (generate-string
                       (for [input inputs]
                         {:method "POST"
                          :relative_url
                          (relative-url {:url (str cdn-prefix "/" (:url input))})}))}}
        fb-result (map (comp #(parse-string % keyword) :body)
                       (fb/body
                        (http/post "https://graph.facebook.com" opts)))]
    (map (fn [m {fb-id :id}]
           (assoc m :facebook-id fb-id, :album-id album-id))
         inputs fb-result)))

(s/defn link-photos! :- [LinkedInput]
  "Uploads the supplied photos to Facebook and adds the facebook id
  and album id into the photo input. These photos can then be added to
  the database."
  [album-id :- ps/AlbumID
   cdn-prefix :- s/Str
   inputs :- [ps/PhotoInput]]
  (let [link (if (conf/prod?)
               link-photos-prod
               link-photos-dev)]
    (link album-id cdn-prefix inputs)))

(s/defn get-photo
  "retrieve photo info from Facebook. See
  https://developers.facebook.com/docs/graph-api/reference/photo/ for
  return schema."
  [photo-id :- ps/FacebookID]
  (fb/api-get photo-id
              {:params
               {:oauth-token
                (fb/server-secret)}
               :query-params {}}))

;; ### Tagging

(s/defn tag-photos-prod :- [{:success s/Bool}]
  "Tags all photos on Facebook. Returns a sequence of tag responses."
  [tags :- [{:photo-fb-id ps/FacebookID
             :user-fb-ids [(s/maybe s/Str)]}]]
  (let [opts {:query-params
              {:access_token (:page-admin-token
                              (conf/facebook-config))
               :batch (generate-string
                       (for [{:keys [photo-fb-id user-fb-ids]} tags
                             id user-fb-ids :when id]
                         {:method "POST"
                          :relative_url (str photo-fb-id "/tags?tag_uid=" id)}))}}]
    (map (comp #(parse-string % keyword) :body)
         (fb/body
          (http/post "https://graph.facebook.com" opts)))))

;; ### Albums

(s/defn create-album! :- {:id s/Str}
  "Creates a new facebook album and returns the ID of that album. In
  dev mode, just returns a fake UUID.

  Reference: https://developers.facebook.com/docs/graph-api/reference/page/albums"
  [m :- {:name s/Str
         :location s/Str}]
  (if (conf/prod?)
    (let [{:keys [page-id page-admin-token]} (conf/facebook-config)]
      (fb/api-post
       (str page-id "/albums")
       {:params {:oauth-token (fb/server-secret)}
        :query-params (assoc m :access_token page-admin-token)}))
    {:id (str (UUID/randomUUID))}))

(s/defn album-url :- s/Str
  "Returns the URL for the supplied album ID."
  [album-id :- s/Str]
  (:link (fb/api-get
          album-id
          {:query-params {:access_token
                          (:page-admin-token
                           (conf/facebook-config))}})))
