(ns hub.photo.client
  (:require [clj-time.coerce :refer [to-long]]
            [clj-time.core :as ct]
            [clojure.string :as st]
            [schema.core :as s]
            [hub.photo.facebook :as fb]
            [hub.photo.schema :as ps]
            [hub.photo.service :as photo]
            [hub.photo.setup :refer [setup!]]
            [hub.util.api :as ua]
            [hub.util.rethink :as ur :refer [RethinkSpec]]
            [rethinkdb.query :as r]
            [rethinkdb.query-builder :as qb]
            [taoensso.timbre :as timbre])
  (:import [com.stuartsierra.component Lifecycle]))

;; ## Create

(declare register-bib-mapping!)

(s/defn create! :- (s/either [ps/ID] {:error s/Str})
  "Generates a bunch of photo instances for the supplied race off of
  the supplied photo URLs. Does not upload the photo to FB."
  [race-id :- ps/RaceID
   photos :- [ps/PhotoInput]]
  (timbre/info {:service "photo"
                :api-call "create!"
                :args {:race-id race-id
                       :photos photos}})
  (if-let [info (photo/get-race-info race-id)]
    (:generated_keys
     (photo/run-photo
      (r/insert
       (map (fn [{:keys [url location filename timestamp dimensions]}]
              (merge
               {:race-id race-id
                :photo-url url
                :tags []
                :timestamp (or timestamp (to-long (ct/now)))}
               (when filename {:filename filename})
               (when location {:location location})
               (when dimensions {:dimensions dimensions})))
            photos))))
    {:error (str race-id " is missing stored information.")}))

;; ## Non-Photo Updates

(s/defn set-user-info! :- s/Bool
  "Registers privacy settings or facebook info for the specified
  user."
  [user-id :- ps/UserID
   opts :-  {(s/optional-key :privacy-level) ps/Privacy
             (s/optional-key :facebook-id) s/Str}]
  (try (photo/set-user-info! user-id opts)
       true
       (catch Exception _ false)))

(s/defn set-privacy! :- s/Bool
  "Registers a default privacy setting for the specified user. All new
  photos tagged will show up with this privacy setting across ALL
  races."
  [m :- {:user-id ps/UserID, :privacy-level ps/Privacy}]
  (set-user-info!
   (:user-id m)
   {:privacy-level (:privacy-level m)}))

(s/defn set-facebook-id! :- s/Bool
  "Registers a default privacy setting for the specified user. All new
  photos tagged will show up with this privacy setting across ALL
  races."
  [m :- {:user-id ps/UserID, :facebook-id s/Str}]
  (set-user-info!
   (:user-id m)
   {:facebook-id (:facebook-id m)}))

(declare photos-by-races)

(s/defn register-bib-mapping! :- s/Bool
  "For the supplied race ID, registers a mapping of bib number to user
  ID. Optionally set a default privacy for the user; if the user has a
  custom default privacy setting registered, the one supplied here
  will have no effect."
  [race-id :- ps/RaceID
   mapping :- {ps/Bib {:user-id ps/UserID
                       (s/optional-key :default-privacy) ps/Privacy}}]
  (timbre/info {:service "photo"
                :api-call "register-bib-mapping!"
                :args {:race-id race-id
                       :mapping mapping}})
  (try (if-let [existing (photo/get-race-info race-id)]
         (photo/run
           (r/table photo/race-info)
           (r/replace (assoc existing :mapping mapping)))
         (photo/run
           (r/table photo/race-info)
           (r/insert {:mapping mapping
                      :race-id race-id})))
       true
       (catch Exception _ false)))

(s/defn initialize-album! :- (s/maybe ps/AlbumID)
  "Creates a Facebook album for the given race. If an album already
  exists, it will not create a new one or overwrite the previous
  one. If there are previous bib mappings, they will remain
  untouched."
  [race-id :- ps/RaceID
   location :- (s/named s/Str "Race Location for FB Album")
   title :- (s/named s/Str "Race title for FB display.")
   fb-page-id :- s/Str
   access-token :- s/Str]
  (timbre/info {:service "photo"
                :api-call "initialize-album!"
                :args {:race-id race-id
                       :fb-page-id fb-page-id
                       :access-token access-token
                       :location location
                       :title title}})
  (try (let [existing (photo/get-race-info race-id)
             album-id (or ;;migrate :album-id + deprecate
                       (:album-id existing)
                       (get (:albums existing) fb-page-id)
                       (:id (fb/create-album!
                             {:name title
                              :location location
                              :page-id fb-page-id
                              :access-token access-token})))]
         (if existing
           (photo/run
             (r/table photo/race-info)
             (r/replace
              (assoc existing :album-id album-id)))
           (photo/run
             (r/table photo/race-info)
             (r/insert {:mapping {}
                        :race-id race-id
                        :album-id album-id})))
         album-id)
       (catch Exception _ false)))

(s/defn captions-for-photos :- {ps/ID s/Str}
  "Takes in a collection of unhydrated photos (no :user-ids, just
  bibs). Returns a map of photo-id to caption string. A caption string
  for a photo with Tim and Dave might look like: 'Dave
  Petrovics (#12), Tim Hornsby (#32)'."
  [photos :- [ps/Photo]
   user-id->full-name :- {s/Str s/Str}]
  (let [photo-caption (fn [{:keys [id tags]}]
                        (let [user-ids->num (into {}
                                                  (map (fn [{:keys [user-tag bib] :as tag}]
                                                         [(:user-id user-tag) bib])
                                                       tags))
                              caption (st/join ", " (map (fn [[user-id num]]
                                                           (str (user-id->full-name user-id)
                                                                " (#" num ")"))
                                                         user-ids->num))]
                          [id caption]))]
    (->>  (map photo-caption (photo/hydrate-users photos))
          (into {}))))

(s/defn upload-album-to-fb! :- s/Bool
  "Uploads all photos in the given album to Facebook. Call this after
  the album has already been initialized, and bib mappings are up to
  date. Call this with a user-id->full-name mapping so that captions
  can be uploaded with the photos."
  [race-id :- ps/RaceID
   cdn-prefix :- s/Str
   user-id->full-name :- {s/Str s/Str}
   fb-page-id :- s/Str
   access-token :- s/Str]
  (timbre/info {:service "photo"
                :api-call "upload-album-to-fb!"
                :args {:race-id race-id
                       :cdn-prefix cdn-prefix
                       :user-id->full-name user-id->full-name}})
  (if-let [{:keys [album-id bib-mapping] :as race-info} (photo/get-race-info race-id)]
    (let [race-photos (get (photos-by-races [race-id]) race-id)
          photo-id->caption (captions-for-photos race-photos user-id->full-name)]
      (doall (map #(photo/run-photo
                    (r/get (:id %))
                    (r/update (select-keys % [:facebook-id :album-id])))
                  (fb/link-photos! album-id
                                   cdn-prefix
                                   photo-id->caption
                                   race-photos
                                   fb-page-id
                                   access-token)))
      true)
    false))

;; ## Get

(s/defn multiget :- {ps/ID ps/Photo}
  "Multiget, same as it always works."
  [ids :- [ps/ID]]
  (photo/multiget ids))

(def get-photo
  (photo/multiget->get multiget))

(s/defn get-race-info :- ps/RaceInfo
  [race-id :- ps/RaceID]
  (timbre/info {:service "photo"
                :api-call "get-race-info"
                :args {:race-id race-id}})
  (photo/get-race-info race-id))

(s/defn photos-by-races :- {ps/RaceID [ps/Photo]}
  "Accepts a race ID and returns a sequence of photos for that race."
  [race-ids :- [ps/RaceID]]
  (if-let [ids (not-empty (distinct race-ids))]
    (->> (photo/run-photo
          (r/get-all ids {:index "by-race-id"})
          (r/coerce-to "ARRAY"))
         (photo/hydrate-users)
         (group-by :race-id))
    {}))

(s/defn photos-by-bib :- [ps/Photo]
  "Accepts a race ID and a bib number and returns a sequence of photos
  for that bib number at that race."
  [race-id :- ps/RaceID bib-id :- ps/Bib]
  (photo/hydrate-users
   (photo/run-photo
    (r/get-all [[race-id bib-id]] {:index "by-bib-id"})
    (r/coerce-to "ARRAY"))))

(s/defn photos-by-user :- {ps/RaceID [ps/Photo]}
  "Accepts a series of race IDs and a set of user IDs and returns a
  map of RaceID to a sequence of photos that any of the supplied users
  appear in (alongside the privacy level)."
  [race-ids :- [ps/RaceID] user-ids :- #{ps/UserID}]
  (->> (photos-by-races race-ids)
       (mapcat (fn [[race-id photos]]
                 (let [filtered (filter
                                 (fn [photo]
                                   (some (comp user-ids :user-id :user-tag)
                                         (:tags photo)))
                                 photos)]
                   (if (not-empty filtered)
                     [[race-id filtered]]))))
       (into {})))

;; Photo Updates

(letfn [(if-photo-exists [id f]
          (if-let [photo (photo/run-photo (r/get id))]
            (f photo)
            {:error
             {:type "photo_doesnt_exist"
              :message (format "Photo with id %s doesn't exist." id)}}))]

  (s/defn geotag! :- (s/either ps/Photo ua/Err)
    "Applies the supplied geotag to the supplied photo."
    [id :- ps/ID geotag :- ps/GeoTag]
    (if-photo-exists
     id
     (fn [photo]
       (let [updated (assoc photo :location geotag)]
         (photo/run-photo
          (r/replace updated))
         (first (photo/hydrate-users [updated]))))))

  (s/defn tag! :- (s/either ps/Photo ua/Err)
    "Marks a tag for the supplied photo ID. If you supply one of
    x-offset and y-offset, the default is zero."
    [photo-id :- ps/ID
     tags :- [{:bib ps/Bib
               (s/optional-key :x-offset) s/Num
               (s/optional-key :y-offset) s/Num}]]
    (if-photo-exists
     photo-id
     (fn [photo]
       (let [tags (map (fn [{:keys [bib x-offset y-offset]}]
                         (merge {:bib bib}
                                (when (or x-offset y-offset)
                                  {:coords {:x-offset (or x-offset 0)
                                            :y-offset (or y-offset 0)}})))
                       tags)
             updated (assoc photo :tags tags)]
         (photo/run-photo
          (r/replace updated))
         (photo/tag-facebook-photos! [updated])
         updated))))

  (s/defn delete!
    "Deletes a photo from the photo table."
    [photo-id :- ps/ID]
    (-> (photo/run (r/table photo/photo-table) (r/get photo-id) r/delete)
        :deleted
        boolean))

  (s/defn delete-race-photos!
    "Deletes all photos for the given race. Does not delete the race info table."
    [race-id :- s/Str]
    (photo/run-photo
     (r/get-all [race-id] {:index "by-race-id"})
     (r/delete)))

  (s/defn delete-race-info!
    [race-id :- s/Str]
    (photo/run
      (r/table photo/race-info)
      (r/get-all [race-id] {:index "by-race-id"})
      r/delete)))


(s/defn merge-users! :- (s/either s/Bool ua/Err)
  "Merges the sub-id user into the primary-id user. Returns the merged
  user. Errors if either user doesn't exist.

  This can be used to merge full users, OR to merge pending users
  together or into full users. Can't merge full users into a pending
  user."
  [primary-id :- ps/UserID
   sub-id :- ps/UserID]
  (let [affected-mappings (photo/run
                            (r/table photo/race-info)
                            (r/get-all [sub-id] {:index "by-user-id"})
                            (r/coerce-to "ARRAY"))
        user-id->info (photo/multiget-user-info [primary-id sub-id])
        updated (for [m affected-mappings]
                  (update m :mapping
                          (fn [mapping]
                            (photo/map-values
                             (fn [{:keys [user-id] :as user-m}]
                               (if (= user-id sub-id)
                                 (assoc user-m :user-id primary-id)
                                 user-m))
                             mapping))))]
    (try (let [pri (user-id->info primary-id)
               sub (user-id->info sub-id)]
           (cond (and pri sub)
                 (do (photo/run
                       (r/table photo/privacy-table)
                       (r/delete sub))
                     (photo/run
                       (r/table photo/privacy-table)
                       (r/replace (merge sub pri))))
                 pri nil
                 sub (photo/run
                       (r/table photo/privacy-table)
                       (r/replace (assoc sub :user-id primary-id)))))
         (photo/run
           (r/table photo/race-info)
           (r/insert updated {:conflict "replace"}))
         true
         (catch Exception _ false))))

;; ## API Initialization

(s/defn photo-client :- Lifecycle
  "Client for the photo service."
  ([]
   (photo-client {:rethink {:mode :live
                            :db "racehub"
                            :spec {}}}))
  ([opts :- {:rethink {:mode (s/enum :test :live)
                       :db s/Str
                       :spec RethinkSpec}}]
   (let [{:keys [mode db spec]} (:rethink opts)]
     (ur/rethinkdb mode db spec setup!))))
