(ns hub.photo.service
  "Photo Service implementation details."
  (:require [com.stuartsierra.component :as c]
            [hub.photo.config :as conf]
            [hub.photo.schema :as ps]
            [hub.util.rethink :as ur]
            [rethinkdb.query :as r]
            [rethinkdb.net :as n]
            [schema.core :as s])
  (:import [com.stuartsierra.component Lifecycle]
           [rethinkdb.net Cursor]))

(def photo-table "photo")
(def privacy-table "photo_privacy")
(def bib-mapping "bib_mapping")

(defmacro run-photo
  "Run a command on the photo table."
  [& forms]
  `(ur/run
     (-> (r/table photo-table)
         ~@forms)))

(defmacro run
  "Run a command on the configured database."
  [& forms]
  `(ur/run
     (-> (r/db (ur/db-name))
         ~@forms)))

(defn map-values [f m]
  (into {} (for [[k v] m]
             [k (f v)])))

(def multiget->get
  "Takes a function of [id] -> {id item} and returns a function of id
  -> (s/maybe item)."
  (fn [f] (fn [x] (second (first (f [x]))))))

(s/defn multiget-privacy-info :- {ps/UserID ps/UserTagDoc}
  [ids :- [ps/UserID]]
  (let [docs (if-let [ids (not-empty (distinct ids))]
               (run
                 (r/table privacy-table)
                 (r/get-all ids {:index "by-user-id"}))
               [])]
    (into {} (map (fn [m] [(:user-id m) m]) docs))))

(def get-privacy-info
  "If the user has privacy settings registered, returns those
  settings."
  (multiget->get multiget-privacy-info))

(s/defn get-mappings :- {s/Str ps/BibMapping}
  "Multiget for bib mappings. Returns a map of race-id -> BibMapping."
  [race-ids :- [s/Str]]
  (if-let [ids (not-empty (distinct race-ids))]
    (let [mappings (->> (run
                          (r/table bib-mapping)
                          (r/get-all ids {:index "by-race-id"}))
                        (map (fn [doc]
                               (letfn [(string-keys [m]
                                         (into {} (map (fn [[k v]] [(name k) v]) m)))]
                                 (update doc :mapping string-keys)))))]
      (into {} (map (fn [m] [(:race-id m) m]) mappings)))
    {}))

(def get-bib-mapping
  "If the user has privacy settings registered, returns those
  settings."
  (multiget->get get-mappings))

(s/defn hydrate-users :- [ps/Photo]
  "Fills in the UserTag field on the tags in each photo."
  [photos :- [ps/Photo]]
  (let [race-ids (map :race-id photos)
        race-id->mapping (get-mappings race-ids)
        tag-user (fn [mapping {:keys [bib] :as tag}]
                   (merge tag (when-let [user-m (get mapping bib)]
                                {:user-tag {:user-id (:user-id user-m)}})))
        user-tags (fn [photo]
                    (if-let [{:keys [mapping]} (race-id->mapping (:race-id photo))]
                      (let [tag-fn (fn [tags]
                                     (map (partial tag-user mapping) tags))]
                        (update photo :tags tag-fn))
                      photo))
        with-users (map user-tags photos)
        user-ids (mapcat (fn [photo]
                           (->> (:tags photo)
                                (map (comp :user-id :user-tag))
                                (remove nil?)))
                         with-users)
        user-id->privacy (multiget-privacy-info user-ids)
        add-privacy (fn [photo {:keys [bib user-tag] :as tag}]
                      (if-not user-tag
                        tag
                        (let [default (-> (:race-id photo)
                                          (race-id->mapping)
                                          :mapping
                                          (get bib)
                                          :default-privacy)]
                          (update tag :user-tag
                                  assoc
                                  :privacy-level
                                  (or (user-id->privacy (:user-id user-tag))
                                      default
                                      (conf/default-privacy))))))]
    (map (fn [photo]
           (update photo :tags (partial map (partial add-privacy photo))))
         with-users)))

(s/defn multiget :- {ps/ID ps/Photo}
  "Same as the client's multiget, but handles hydrating the bib
  mappings."
  [ids :- [ps/ID]]
  (if-let [ids (not-empty (distinct ids))]
    (->> (run-photo (ur/get-all ids))
         (hydrate-users)
         (map (juxt :id identity))
         (into {}))
    {}))
