(ns hub.photo.client
  (:require [clj-time.coerce :refer [to-long]]
            [clj-time.core :as ct]
            [schema.core :as s]
            [hub.photo.schema :as ps]
            [hub.photo.service :as photo]
            [hub.photo.setup :refer [setup!]]
            [hub.util.rethink :as ur :refer [RethinkSpec]]
            [rethinkdb.query :as r]
            [rethinkdb.query-builder :as qb])
  (:import [com.stuartsierra.component Lifecycle]))

;; # API

(s/defschema PhotoInput
  {:url ps/PhotoURL
   (s/optional-key :location) ps/GeoTag
   (s/optional-key :dimensions) ps/Dims
   (s/optional-key :timestamp) s/Int})

;; ## Create

(s/defn create! :- [ps/ID]
  "Generates a bunch of photo instances for the supplied race off of
  the supplied photo URLs."
  [race-id :- s/Str
   photos :- [PhotoInput]]
  (:generated_keys
   (photo/run-photo
    (r/insert
     (map (fn [{:keys [url location timestamp dimensions]}]
            (merge
             {:race-id race-id
              :photo-url url
              :tags []
              :timestamp (or timestamp (to-long (ct/now)))}
             (when location {:location location})
             (when dimensions {:dimensions dimensions})))
          photos)))))

;; ## Update

;; TODO: Expand this to work with multiple items.
(s/defn set-privacy! :- s/Bool
  "Registers a default privacy setting for the specified user. All new
  photos tagged will show up with this privacy setting across ALL
  races."
  [m :- {:user-id ps/UserID, :privacy-level ps/Privacy}]
  (try (let [table (-> (r/db (ur/db-name))
                       (r/table photo/privacy-table))
             result (photo/run
                      (-> (r/table photo/privacy-table)
                          (r/get-all [(:user-id m)] {:index "by-user-id"})
                          (r/is-empty)
                          (r/branch
                           (r/insert table m)
                           (r/get-all table [(:user-id m)] {:index "by-user-id"}))))]
         true)
       (catch Exception _ false)))

(s/defn register-bib-mapping! :- s/Bool
  "For the supplied race ID, registers a mapping of bib number to user
  ID. Optionally set a default privacy for the user; if the user has a
  custom default privacy setting registered, the one supplied here
  will have no effect.

  This call will potentially trigger user tag events on the event
  stream (if there are photos with tagged bibs and no linked photos,
  for example)."
  [race-id :- s/Str
   m :- {ps/Bib {:user-id ps/UserID
                 (s/optional-key :default-privacy) ps/Privacy}}])

(s/defn geotag! :- ps/Photo
  "Applies the supplied geotag to the supplied photo."
  [id :- ps/ID geotag :- ps/GeoTag])

(s/defn tag! :- ps/Photo
  "Marks a tag for the supplied photo ID."
  [photo-id :- ps/ID
   tags :- [{:bib ps/Bib
             (s/optional-key :x-offset) s/Num
             (s/optional-key :y-offset) s/Num}]])

;; ## Get

(comment
  ;; Weird bug here:
  (photo/run-photo
   (r/get-all ["2b40e32f-9192-41f8-9c1c-4ee6d1dbcdb1" "sdfsdf"])))

(s/defn multiget :- {ps/ID ps/Photo}
  "Multiget, same as it always works."
  [ids :- #{ps/ID}]
  (->> (photo/run-photo (r/get-all ids))
       (map (juxt :id identity))
       (into {} )))

(s/defn photos-by-user
  :- {ps/UserID {ps/RaceID
                 [{:photo ps/Photo
                   :privacy-level ps/Privacy}]}}
  "Accepts a set of user IDs and returns a map of user id to a map of
  RaceID to a sequence of photos that the user appears in (alongside
  the privacy level)."
  [user-ids :- #{ps/UserID}])

(s/defn photos-by-race :- [ps/Photo]
  "Accepts a race ID and returns a sequence of photos for that race."
  [race-id :- ps/RaceID]
  (photo/run-photo
   (r/get-all [race-id] {:index "by-race-id"})))

(s/defn photos-by-bib :- [ps/Photo]
  "Accepts a race ID and a bib number and returns a sequence of photos
  for that bib number at that race."
  [race-id bib-id :- ps/Bib]
  (photo/run-photo
   (r/get-all [[race-id bib-id]] {:index "by-bib-id"})))

;; TODO: user merges! We'll need this bigtime for pending user merges.

;; ## API Initialization

(s/defn photo-client :- Lifecycle
  "Client for the photo service."
  ([]
   (photo-client {:rethink {:mode :live
                            :db-name "racehub"
                            :spec {}}}))
  ([opts :- {:rethink {:mode (s/enum :test :live)
                       :db-name s/Str
                       :spec RethinkSpec}}]
   (let [{:keys [mode db-name spec]} (:rethink opts)]
     (ur/rethinkdb mode db-name spec setup!))))
