(ns hub.notification.client
  "This is probably best implemented with Redis and a set."
  (:require [com.stuartsierra.component :as c]
            [finagle-clojure.futures :as f]
            [hub.notification.service :as n]
            [schema.core :as s]
            [taoensso.carmine :as car])
  (:import [com.stuartsierra.component Lifecycle]))

(def ID (s/named s/Str "User ID."))
(def Notification (s/named s/Str "Notification Type"))
(defn Future [s] s/Any)

;; ## Create / Update

;; needs to take the "OK" return shit out of there
(s/defn update! :- (Future {Notification s/Bool})
  "Updates the supplied notification values. Returns the total map of
  notification opt-in settings for the user."
  [id :- ID m :- {Notification s/Bool}]
  (-> (n/wcar (when-not (empty? m)
                (apply car/hmset id (apply concat m)))
              (car/hgetall id))
      (f/value)
      (f/map* (comp (partial apply hash-map) second))))

(s/defn opt-in! :- (Future {Notification s/Bool})
  "Takes in a user and a sequence of choices and records them all as
  `true`, replacing others. Returns the total map of notification
  opt-in settings for the user."
  [id :- ID choices :- [Notification]]
  (let [kvs (interleave choices (repeat true))]
    (n/wcar (car/del id)
            (when-not (empty? choices)
              (apply car/hmset id kvs)))
    (f/value (apply hash-map kvs))))

;; ## Get

(s/defn opt-ins :- (Future {Notification s/Bool})
  "Returns the total map of notification opt-in settings for the
   user (or an error if the user doesn't exist)."
  [id :- ID]
  (-> (f/value (n/wcar (car/hgetall id)))
      (f/map* (partial apply hash-map))))

(s/defn opted-in? :- (Future s/Bool)
  "Returns true if the user opted in for that particular setting,
  false otherwise."
  [id :- ID notification :- Notification]
  (-> (n/wcar (car/hget id notification))
      (f/value)
      (f/map* boolean)))

;; ## Service Startup

(s/defschema RedisSpec
  {(s/optional-key :host) s/Str
   (s/optional-key :port) s/Int
   (s/optional-key :password) s/Str
   (s/optional-key :timeout-ms) s/Int
   (s/optional-key :db) s/Int})

(s/defn notification-client :- Lifecycle
  "Client for the notification service."
  ([] (notification-client {}))
  ([spec :- RedisSpec]
   (reify c/Lifecycle
     (start [_] (reset! n/conf {:pool {} :spec spec}))
     (stop [_] (reset! n/conf nil)))))
