(ns pca.core
  (:require [clojure.core.matrix :as mat]
            [clojure.core.matrix.linear :as lin]
            [clatrix.core :as ctx]))

(mat/set-current-implementation :clatrix);;Currently only implementation to provide proper eigen

;;Input can be matrix or lazy seq
;;Implemented from http://rebcabin.github.io/blog/2013/01/22/covariance-matrices/
(defn covariance [m]
  "Incremental calculation of covariance m can be a matrix or a lazyseq."
  (let [vec-w (count (first m))
        sum (mat/zero-vector vec-w)
        op-sum (mat/zero-matrix vec-w vec-w)]
    (loop [coll m i 0]
      (if (empty? coll)
        (mat/sub! (mat/div op-sum i) (mat/square (mat/div sum i)));;Calculate covariance
        (let [v (first m)]
          (mat/add! sum v)
          (mat/add! op-sum (mat/outer-product v v))
          (recur (rest coll) (inc i)))))))

;;Implemented from this http://sebastianraschka.com/Articles/2014_pca_step_by_step.html#transforming-the-samples-onto-the-new-subspace
(defn principal-components [covar n]
  (let [eigen-decomp (ctx/eigen (mat/matrix covar));;Using clatrix eigen because it is implemented properly
        value-vec-pairs (map (fn [a b] [a b]) (:values eigen-decomp) (:vectors eigen-decomp))
        sorted-eigens (reverse (sort-by first value-vec-pairs))]
    (map (fn [pair i] (vec (second pair))) sorted-eigens (range 0 n))))

;;Basically dot product
(defn compress [components v]
  (reduce mat/add (mat/transpose (mat/mul components v))))

(defn decompress [components compressed-v]
  (compress (mat/transpose components) compressed-v))
