(ns cirque.scheduler
  "Looks for available delayed jobs and enqueues them as regular ones."
  (:require [cirque.resque :as resque]
            [cirque.redis :as redis]
            [clojure.data.json :as json])
  (:use [cirque.util]))

(def should-poll? (ref true))

(def config (atom {:poll-interval (* 1 1000)
                   :scheduler-enabled? false}))

(declare poll
         enqueue-jobs-for-timestamp
         transfer)

(defn configure [c]
  (swap! config merge c))

(defn stop []
  "stops polling"
  (dosync (ref-set should-poll? false)))

(defn start []
  "start polling the delayed_queue_schedule for jobs to be enqueued"
  (if (:scheduler-enabled? @config)
    (do (dosync (ref-set should-poll? true))
        (.start (Thread. poll))
        (.addShutdownHook (Runtime/getRuntime)
                          (Thread. stop)))))

(defn poll []
  (if @should-poll?
    (let [next-timestamp (resque/next-delayed-timestamp)]
      (if next-timestamp (enqueue-jobs-for-timestamp next-timestamp))
      (Thread/sleep (:poll-interval @config))
      (recur))))

(defn enqueue-jobs-for-timestamp [timestamp]
  "Works all delayed jobs for a timestamp"
  (loop [job (resque/pop-job timestamp)]
    (if job
      (do
        (transfer job timestamp)
        (recur (resque/pop-job timestamp))))))

(defn transfer [job timestamp]
  "Takes a delayed job and enqueues it to be worked ASAP"
  (let [job (json/read-str job)
        {:strs [queue class args]} job]
  (apply (partial resque/enqueue queue class) args)))
