(ns cirque.resque
  "This namespace provides a high level API for performing
   resque operations on the redis cluster."
  (:import [java.util Date])
  (:require [cirque.redis :as redis]
            [clojure.data.json :as json]
            [cirque.worker :as worker]))

;; private api
(declare -namespace-key
         -full-queue-name
         -format-error
         -dequeue-randomized)

;;
;; public api
;;

(def config (atom {:namespace "resque"
                   :error-handler nil}))

(defn configure [c]
  (swap! config merge c))

(defn enqueue [queue worker-name & args]
  (redis/sadd (-namespace-key "queues") queue)
  (redis/rpush (-full-queue-name queue)
               (json/json-str {:class worker-name :args args})))


(defn enqueue-at [ts queue worker-name & args]
  (let [ts (long (/ ts 1000)) ; Assume input ts is in ms
        delayed-key (-namespace-key (str "delayed:" ts))
        encoded-task (json/json-str {:queue queue :class worker-name :args args})
        timestamps-key (-namespace-key (str "timestamps:" encoded-task))
        z-queue-key (-namespace-key "delayed_queue_schedule")]
  ; enqueue the encoded job into a list per timestmp to be popped and workered later
  (redis/rpush delayed-key encoded-task)

  ; save the job + args into a set so that it can be checked by plugins
  (redis/sadd timestamps-key delayed-key)

  ; and the timestamp in question to a zset to the scheduler will know which timestamps have data to work
  (redis/zadd z-queue-key (double ts) (str ts))))

(defn dequeue [queues]
  "Randomizes the list of queues. Then returns the first queue that contains a job.
   Returns a hash of: {:queue \"queue-name\" :data {...}} or nil"
  (let [msg (-dequeue-randomized queues)]
    (if msg
      (let [{:keys [class args]} (json/read-json (:data msg))]
        (assoc msg :func class :args args)))))

(defn report-error [result]
  (let [error (-format-error result)
        handle (:error-handler @config)]
    (redis/rpush (-namespace-key "failed") (json/json-str error))
    (when handle
      (handle error))))

(defn register [queues]
  (let [worker-name (worker/name queues)
        worker-started-key (str "worker:" worker-name ":started")
        time (format "%1$ta %1$tb %1$td %1$tk:%1$tM:%1$tS %1$tz %1$tY" (Date.))]
    (redis/sadd (-namespace-key "workers") worker-name)
    (redis/set (-namespace-key worker-started-key) time)))

(defn unregister [queues]
  (let [worker-name (worker/name queues)
        keys (redis/keys (str "*" worker-name "*"))
        workers-set (-namespace-key "workers")]
    (redis/del worker-name)
    (redis/srem workers-set worker-name)
    (if (empty? (redis/smembers workers-set))
      (redis/del workers-set))
    (doseq [key keys]
      (redis/del key))))

(defn next-delayed-timestamp []
  "Returns the next timestamp to be worked."
  (->
    (-namespace-key "delayed_queue_schedule")
    (redis/zrange-by-score 0.0 (double (/ (System/currentTimeMillis) 1000)) 0 1)
    (first)
    (#(if % (.getElement %)))))

(defn pop-job [timestamp]
  "Pops the next job for a timestamp.
   Also cleans any redis timestamps."
  (let [job-key (-namespace-key (str "delayed:" timestamp))
        job (redis/lpop job-key)]
    ; CLEANUP TIME
    (redis/srem (-namespace-key (str "timestamps:" job)) job-key)
    (redis/del (-namespace-key (str "delayed:" timestamp)))
    (redis/zrem (-namespace-key "delayed_queue_schedule") (str timestamp))
    ; return the popped job
    job))

;;
;; private
;;

(defn -namespace-key [key]
  (str (:namespace @config) ":" key))

(defn -full-queue-name [name]
  (-namespace-key (str "queue:" name)))

(defn -format-error [result]
  (let [exception (:exception result)
        stacktrace (map #(.toString %) (.getStackTrace exception))
        exception-class (-> exception (.getClass) (.getName))]
    {:failed_at (format "%1$tY/%1$tm/%1$td %1$tk:%1$tM:%1$tS" (Date.))
     :payload (select-keys result [:job :class :args])
     :exception exception-class
     :error (or (.getMessage exception) "(null)")
     :backtrace stacktrace
     :worker (apply str (interpose ":" (reverse (.split (.getName (java.lang.management.ManagementFactory/getRuntimeMXBean)) "@"))))
     :queue (:queue result)}))

(defn -dequeue-randomized [queues]
  "Randomizes the list of queues. Then returns the first queue that contains a job"
  (loop [qs (shuffle queues)]
    (let [q (first qs)
          nsq (-full-queue-name q)
          job (redis/lpop nsq)]
      (cond
       (empty? qs) nil
       job {:queue q :data job}
       :else (recur (rest qs))))))
