(ns cirque.core
  (:use [cirque.util :only [filter-map]])
  (:require [cirque.resque :as resque]
            [cirque.redis :as redis]
            [cirque.supervisor :as supervisor]
            [cirque.scheduler :as scheduler]))


(defn configure [c]
  "set configuration parameters. Available keys:
     :host
     :port
     :password
     :uri
     :database
     :timeout
     :namespace
     :max-shutdown-wait
     :poll-interval
     :max-workers
     :error-handler
     :scheduler-enabled?"
  (let [redis-keys (keys @redis/config)
        resque-keys (keys @resque/config)
        super-keys (keys @supervisor/config)
        sched-keys (keys @scheduler/config)

        redis-map (filter-map c redis-keys)
        resque-map (filter-map c resque-keys)
        super-map (filter-map c super-keys)
        sched-map (filter-map c sched-keys)]

    (redis/configure redis-map)
    (resque/configure resque-map)
    (supervisor/configure super-map)
    (scheduler/configure sched-map)))


(defn enqueue-at [ts queue worker-name & args]
  "create a new delayed resque job
     ts: unix timestamp at which the job should be run
     queue: name of the queue (does not start with resque:queue)
     worker-name: fully-qualified function name (ex. clojure.core/str in clojure, MyWorker in ruby)
     args: data to be sent as args to the worker. must be able to be serialized to json"
  (apply resque/enqueue-at ts queue worker-name args))

(defn enqueue [queue worker-name & args]
  "create a new resque job
     queue: name of the queue (does not start with resque:queue)
     worker-name: fully-qualified function name (ex. clojure.core/str in clojure, MyWorker in ruby)
     args: data to be sent as args to the worker. must be able to be serialized to json"
  (apply resque/enqueue queue worker-name args))

(defn start [queues]
  "start listening for jobs on queues (vector).
   runs scheduler if scheduler-enabled? is set to true."
  (supervisor/start queues)
  (scheduler/start))

(defn stop []
  "stops polling queues. waits for all workers to complete current job"
  (supervisor/stop)
  (scheduler/stop))
