(ns exchange.web
  (:require [compojure.route :refer (not-found)]
            [compojure.core :refer (routes ANY)]
            [ring.middleware.params :as params]
            [ring.middleware.cookies :as cookies]
            [ring.middleware.multipart-params :as mp]
            [ring.adapter.jetty :as ring]
            [clojure.tools.logging :as log]))

;; ============================
;; Ring 中间件
(defn wrap-log
  "包装日志"
  [handler]
  (fn [request]
    (log/debug {:wrap-log {:request (str request)}})
    (let [response (handler request)]
      (log/debug {:wrap-log {:response (str response)}})
      response)))

(defn wrap-exception
  "包装异常处理"
  [handler]
  (fn [request]
    (try
      (handler request)
      (catch Throwable e
        (log/error e {:wrap-exception {:request (str request)}})))))

(defn mk-handler
  "输入成对的vector  每个元素是一对映射的路径和handler函数。
  最后包装成为 ring 的 handler 函数. handler函数参数: 请求数据"
  [route-mapping]
  (log/debug {:mk-handler route-mapping})
  (let [create-route (fn [[r-path p-fn]]
                       (ANY r-path req
                            (p-fn req)))]
    (->> (map create-route route-mapping)
         (into '((not-found "invalid url")))
         (apply routes))))

(defn app
  "将基本处理函数加上默认的中间件包装成为最终的 app"
  [basic-handler]
  (-> basic-handler
      params/wrap-params
      cookies/wrap-cookies
      mp/wrap-multipart-params
      wrap-exception
      wrap-log))

;; ==========================
;; 外部使用的函数
(defn start-web-server
  "启动WEB服务
  route-mapping为包含成对vector的vector
  每个vector元素是一对映射的路径和handler函数
  handler函数参数: req请求数据"
  [web-cfg route-mapping]
  (let [server (-> (mk-handler route-mapping)
                   app
                   (ring/run-jetty web-cfg))]
    (fn [] (.stop server))))

(comment
  (defn handler [req]
    (prn req))
  (def w (start-web-server {:host "0.0.0.0"
                            :port 8765
                            :join? false} [["/log" handler]]))
  (w))