(ns exchange.util
  (:import [java.io PushbackReader])
  (:require [clj-time.coerce :as tc]
            [cheshire.core :as json]
            [clojure.edn]
            [clojure.java.io :as cio]))

(defn map-vals
  "返回用函数更新各value后的map"
  [f m]
  (apply merge (map (fn [[k v]] {k (f v)}) m)))

(defn map-keys
  "返回用函数更新各value后的map"
  [f m]
  (apply merge (map (fn [[k v]] {(f k) v}) m)))

(defn time-str->long
  "将时间字符串转换为对应时区的毫秒时间戳"
  [time-zone s]
  (when-let [utc-long (tc/to-long s)]
    (- utc-long (* time-zone 1000 60 60))))

(defn time-str->local-long
  "将时间字符串转换为北京时区的毫秒时间戳"
  [s]
  (time-str->long 8 s))

(defn now-time
  "当前时间戳"
  []
  (System/currentTimeMillis))

(defn rand-string
  "随机字符串"
  []
  (str (java.util.UUID/randomUUID)))

(defn day-time-period
  "当天的起止时间毫秒数"
  [millis]
  (let [one-day (* 24 60 60 1000)
        day-begin (- millis (rem millis one-day))]
    [day-begin (+ day-begin one-day)]))

(defn append-file
  "将map的内容追加到文件中"
  [file-name map-data]
  (cio/make-parents file-name)
  (with-open [file (cio/writer file-name :append true)]
    (.write file (str map-data "\n"))))

(defn read-file-by-line
  "以行为单位读取文件,保存各行到一个list中"
  [file-name]
  (with-open [rd (cio/reader file-name :encoding "utf-8")]
    (doall (line-seq rd))))

(defn load-edn-config
  "加载edn配置"
  [edn-filename]
  (with-open [in-edn (-> edn-filename
                         cio/resource
                         cio/reader
                         (PushbackReader.))]
    (clojure.edn/read in-edn)))