(ns exchange.system
  (:require [com.stuartsierra.component :as component]
            [clojure.core.async :as a]
            [clojure.data.priority-map :as pm]
            [rmi-service.protocol :as rp]
            [rmi-service.rmi.datacenter]
            [clj-rpc.client :as client]
            [exchange.web :as web]
            [exchange.core :as c]
            [exchange.exchange-process :as ep]
            [exchange.schedule :as es]
            [exchange.log :as log]
            [exchange.util :as u]
            [exchange.db.protocol :as dp]
            [exchange.db.datomic]))

; ==============================================
; 奖券生成
; ==============================================

(defn mk-gen-ticket-fn
  "奖券生成服务接口构造器"
  [time-nodes games thresholds topic db]
  (fn [gamelog]
    (log/debug {:svr_method "add-tickets" :gamelog gamelog})
    (let [tickets (c/log->tickets time-nodes games thresholds topic gamelog)
          results (map #(dp/add-ticket db %) tickets)]
      (dorun (map (fn [ticket result]
                    (when ticket
                      (log/info {:svr_method "add-tickets" :delta ticket :result result})))
                  tickets results)))))

(defrecord 奖券生成 [time-nodes games thresholds topic db]
  component/Lifecycle
  (start [this]
    (assoc this :fn-gen-ticket (mk-gen-ticket-fn time-nodes games thresholds topic db)))
  (stop [this]
    (dissoc this :fn-gen-ticket)))

(defn 生成奖券
  "被外部使用,用于生成对应类型的奖券"
  [奖券生成 log]
  ((:fn-gen-ticket 奖券生成) log))

; ==============================================
; 奖券查询
; ==============================================

(defn mk-ticket-fn
  "奖券查询服务接口构造器"
  [topic ticket-names comp-order db]
  (let [ticket-fn (fn [tickets name]
                    (nth (filter #(= name (:name %)) tickets) 0
                         {:name name :num 0 :exchanged 0 :topic topic}))]
    (fn [username]
      (let [db-tickets (dp/user-tickets db topic username)
            total-tickets (if (= (count ticket-names) (count db-tickets))
                            db-tickets
                            (map (partial ticket-fn db-tickets) ticket-names))
            delta-tickets (c/comp-tickets comp-order total-tickets)]
        (if-not (every? #(zero? (:num %)) delta-tickets)
          (let [result (dp/update-tickets db topic username delta-tickets)]
            (log/info {:svr_method "update-tickets" :origin total-tickets :delta delta-tickets :result result})
            result)
          total-tickets)))))                                ;todo 从小到大输出

(defrecord 奖券查询 [topic ticket-names comp-order db]
  component/Lifecycle
  (start [this]
    (assoc this :fn-ticket (mk-ticket-fn topic ticket-names comp-order db)))
  (stop [this]
    (dissoc this :fn-ticket)))

(defn 查询奖券
  "被外部使用,用于查询用户的奖券信息"
  [奖券查询 username]
  ((:fn-ticket 奖券查询) username))

(defn 查询单个奖券
  "被外部使用,用于查询用户单条奖券信息"
  [奖券查询 username ticket-name]
  (->> (查询奖券 奖券查询 username)
       (filter #(= ticket-name (:name %)))
       first
       :num))

; ==============================================
; 卡密信息
; ==============================================

(defn mk-卡密发放
  "卡密信息构造器"
  [all-cards provided-cards]
  (merge-with #(ep/mk-card-code %1 %2) all-cards provided-cards))

(defn 发放卡密
  "发放一组卡密信息"
  [卡密发放 prize]
  ((卡密发放 prize)))

; ==============================================
; 兑换状态
; ==============================================

(defn 获取奖品
  "被外部使用,用于抢占一个数量的奖品,返回一个函数"
  [兑换状态]
  (:take-prize! 兑换状态))

(defn 重置日奖品兑换数
  "被外部使用,重置每日的兑换累计数"
  [兑换状态]
  ((:reset-day-prizes! 兑换状态)))

(defn 奖品兑换数量
  "被外部使用,返回当前各奖品已兑换的数量"
  [兑换状态]
  ((:exchanged-prizes 兑换状态)))

; ==============================================
; 兑换结果保存
; ==============================================

(defn mk-save-record
  "保存兑换结果服务构造器"
  [file-name db]
  (fn [record]
    (let [result (dp/exchange-ticket db record)]
      (u/append-file file-name record)
      result)))

(defn mk-save-address
  "保存地址服务构造器"
  [file-name db]
  (fn [{:keys [order-id] :as inf}]
    (let [record (dp/user-record db order-id)
          result (cond
                   (nil? record) :no-order
                   (not= :entity (:type record)) :no-need-addr
                   (:provided? record) :addr-repeat
                   :else :ok)]
      (log/debug {:svr_method "save-address" :record record :inf inf})
      (when (= :ok result)
        (let [new-record (-> (select-keys inf [:address :recipient
                                               :phone :postcode])
                             (merge record)
                             (assoc :provided? true))]
          (dp/update-record db new-record)
          (log/info {:svr_method "save-address" :result new-record})
          (u/append-file file-name new-record)))
      result)))

(defrecord 保存服务 [file-name db]
  component/Lifecycle
  (start [this]
    (assoc this :fn-save-address (mk-save-address file-name db)
                :fn-save-record (mk-save-record file-name db)))
  (stop [this]
    (dissoc this :fn-save-address :fn-save-record)))

(defn 保存兑换结果
  "被外部使用,保存兑换结果"
  [保存服务 record]
  ((:fn-save-record 保存服务) record))

(defn 保存实物奖品记录
  "被外部使用,保存地址等信息到指定的兑换记录"
  [保存服务 {:keys [order-id address recipient phone postcode] :as inf}]
  ((:fn-save-address 保存服务) inf))

; ==============================================
; 兑换服务
; ==============================================

(defn mk-exchange-fn
  "兑换服务接口构造器"
  [topic prizes 奖券查询 兑换状态 保存服务 卡密发放 悟性发放 fn-pub]
  (fn [username time prize order-id]
    (let [ticket (get-in prizes [prize :ticket])
          cost (get-in prizes [prize :cost])
          type (get-in prizes [prize :type])
          money (get-in prizes [prize :money])
          exchangeable (查询单个奖券 奖券查询 username ticket)
          pre-record {:order-id order-id
                      :topic    topic
                      :username username
                      :time     time
                      :ticket   ticket
                      :cost     cost
                      :prize    prize
                      :type     type}
          result (ep/exchange-result exchangeable cost
                                     (partial (获取奖品 兑换状态) pre-record))
          prize (when (= :ok result)
                  (case type
                    :card (发放卡密 卡密发放 prize)
                    :money {:add-money money
                            :finance   ((rp/deposite 悟性发放 username money) username)}
                    nil))
          record (merge pre-record prize {:provided? (some? prize)})]
      (log/debug {:svr_method "exchange" :record record :result result})
      (when (= :ok result)
        (let [db-result (保存兑换结果 保存服务 record)]
          (log/info {:svr_method "exchange" :record record :result db-result})
          (fn-pub record)))
      (assoc record :result result))))

(defrecord 兑换服务 [topic prizes 奖券查询 兑换状态 保存服务 卡密发放 悟性发放 fn-pub]
  component/Lifecycle
  (start [this]
    (assoc this :fn-exchange (mk-exchange-fn topic prizes 奖券查询 兑换状态 保存服务 卡密发放 悟性发放 fn-pub)))
  (stop [this]
    (dissoc this :fn-exchange)))

(defn 兑换奖品
  "被外部使用,兑换一个奖品"
  [兑换服务 username time prize order-id]
  ((:fn-exchange 兑换服务) username time prize order-id))

; ==============================================
; 每日重置服务
; ==============================================

(defn reset-daily-prize
  "重置每日"
  [兑换状态]
  (重置日奖品兑换数 兑换状态)
  (log/info {:svr_method "reset-daily-prize" :result (奖品兑换数量 兑换状态)}))

(defrecord 重置日兑换状态 [name hour minute 兑换状态]
  component/Lifecycle
  (start [{:keys [stop-fn] :as this}]
    (if stop-fn
      this
      (do
        (println "start" name)
        (assoc this :stop-fn (es/start-daily-schedule name hour minute #(reset-daily-prize 兑换状态))))))
  (stop [{:keys [stop-fn] :as this}]
    (if stop-fn
      (do
        (println "stop" name)
        (stop-fn)
        (dissoc this :stop-fn))
      this)))

; ==============================================
; 事件通道
; ==============================================

(defn sink
  "对 input-chan 上的每个事件执行函数 f, 返回一个无参数的停止函数."
  [f input-chan]
  (let [cmd-chan (a/chan 1)]
    (a/go-loop []
      (let [[val] (a/alts! [input-chan cmd-chan])]
        (when (and val (not= val ::close))
          (f val)
          (recur))))
    (fn []
      (a/put! cmd-chan ::close))))

(defn tap-ch
  "在多重通道 multi 上注册一个新的通道"
  [multi]
  (let [ch (a/chan (a/dropping-buffer 100))]
    (a/tap multi ch)
    ch))

; ==============================================
; 显示面板
; ==============================================

(defrecord DisplayBoard [name a-obj updater input-ch]
  component/Lifecycle
  (start [{:keys [stop-fn] :as this}]
    (if stop-fn
      this
      (do
        (println "start" name)
        (assoc this :stop-fn (sink (updater a-obj) input-ch)))))
  (stop [{:keys [stop-fn] :as this}]
    (if stop-fn
      (do
        (println "stop" name)
        (stop-fn)
        (dissoc this :stop-fn))
      this)))

; ==============================================
; 兑换排行
; ==============================================

(defn mk-rank-updater
  "排行榜更新"
  [ticket-name a-rank-board]
  (fn [{:keys [username cost ticket] :as record}]
    (log/debug {:svr_method "rank-updater" :record record})
    (when (= ticket-name ticket)
      (swap! a-rank-board update-in [username] (fnil + 0) cost))))

(defn mk-兑换排行
  "排行榜构造函数, 仅仅用在 system 生成中"
  [ch init-data ticket-name]
  (->DisplayBoard "rank server" (atom (into (pm/priority-map-by >) init-data))
                  (partial mk-rank-updater ticket-name) ch))

(defn 显示排行榜
  "被外部使用, 用于显示"
  [排行榜 n]
  (->> 排行榜 :a-obj deref (take n)))

; ==============================================
; 兑换历史
; ==============================================

(defn mk-history-updater
  "历史的更新"
  [max-history prize-filter a-history]
  {:pre [(set? prize-filter)]}
  (fn [{:keys [type] :as record}]
    (log/debug {:svr_method "history-updater" :record record})
    (when (prize-filter type)
      (let [f (fn [history-list]
                (->> (select-keys record [:username :time :cost :prize])
                     (conj history-list)
                     (take max-history)))]
        (swap! a-history f)))))

(defn mk-兑换历史
  "历史构造函数, 仅仅用在 system 生成中"
  [ch max-history prize-filter init-data]
  (let [init-data (map #(select-keys % [:username :time :cost :prize]) init-data)]
    (->DisplayBoard "history server" (atom (apply list init-data))
                    (partial mk-history-updater max-history prize-filter) ch)))

(defn 显示兑换历史
  "被外部使用, 用于显示"
  [历史 start num]
  (->> 历史 :a-obj deref (drop start) (take num)))

(defn 显示历史长度
  "被外部使用, 获取已保存的历史记录长度"
  [历史]
  (->> 历史 :a-obj deref count))

; ==============================================
; logstash 对接服务
; ==============================================

(defn mk-logstash-handler
  [奖券生成]
  (fn [{:keys [body]}]
    (-> body
        (clojure.java.io/reader)
        (cheshire.core/parse-stream)
        (生成奖券 奖券生成))))

(defrecord logstash服务 [name web-cfg url 奖券生成]
  component/Lifecycle
  (start [{:keys [stop-fn] :as this}]
    (if stop-fn
      this
      (do
        (println "start" name)
        (let [logstash-handler (mk-logstash-handler 奖券生成)]
          (assoc this :stop-fn (web/start-web-server web-cfg [[url logstash-handler]]))))))
  (stop [{:keys [stop-fn] :as this}]
    (if stop-fn
      (do
        (println "stop" name)
        (stop-fn)
        (dissoc this :stop-fn))
      this)))

; ==============================================
; 活动 web服务
; ==============================================

(defn attach-info
  "附加信息组合逻辑"
  [msg {:keys [type card code add-money finance
               address express-company express-id]}]
  (let [msg-f (fn [k1 s1 k2 s2]
                (when (or s1 s2)
                  (str (msg k1) s1 (msg k2) s2)))]
    (case type
      :card (msg-f :msg-card-card card
                   :msg-card-code code)
      :entity (or (msg-f :msg-entity-ecom express-company
                         :msg-entity-eid express-id)
                  (msg-f :msg-entity-no-express nil
                         :msg-entity-addr address)
                  (msg :msg-entity-no-addr))
      :money (msg-f :msg-money-add add-money
                    :msg-money-result finance)
      nil)))

(defn mk-web-handler
  [topic web-params time-nodes prizes max-rank msg
   奖券查询 兑换状态 兑换服务 保存服务 兑换历史 兑换排行 db]
  (let [activity-status-fn
        (fn [_ time-fn]
          (let [{:keys [status remain]}
                (c/time->status time-nodes (time-fn))]
            {:result  true
             :msg     :msg-ok
             :status  status
             :remain  remain
             :configs web-params}))
        user-ticket-fn
        (fn [{:keys [username]}]
          (when (string? username)
            (let [transfer (fn [{:keys [name num exchanged]}]
                             {:name      name
                              :number    num
                              :exchanged exchanged})
                  tickets (->> (查询奖券 奖券查询 username)
                               (map transfer))]
              ;(prn tickets)
              {:result  true
               :msg     :msg-ok
               :tickets tickets})))
        user-history-fn
        (fn [{:keys [username query-user]}]
          (when (every? string? [username query-user])
            (let [user username
                  records (->> (dp/user-records db topic query-user)
                               (sort-by (comp - :time)))
                  transfer
                  (fn [{:keys [order-id username time ticket cost
                               prize type provided?] :as record}]
                    (let [{:keys [o h a]}
                          (when (= user query-user)
                            {:o order-id
                             :h (if (= type :entity) provided? true)
                             :a (attach-info msg record)})]
                      {:username       username
                       :time           time
                       :ticket         {:cost cost
                                        :name ticket}
                       :prize          prize
                       :order-id       o
                       :have-post-addr h
                       :attach-info    a}))]
              ;(prn records)
              {:result  true
               :msg     :msg-ok
               :history (map transfer records)})))
        prize-num-fn
        (fn [_]
          (let [exchanged (奖品兑换数量 兑换状态)
                transfer
                (fn [[prize-name {:keys [day-limit all-limit]}]]
                  (let [day-exchanged (get-in exchanged [:day prize-name])
                        all-exchanged (get-in exchanged [:all prize-name])
                        remain-fn (fn [limit exchanged]
                                    (if (= -1 limit)
                                      limit (- limit exchanged)))]
                    {:name         prize-name
                     :day-exchange day-exchanged
                     :all-exchange all-exchanged
                     :day-remain   (remain-fn day-limit day-exchanged)
                     :all-remain   (remain-fn all-limit all-exchanged)}))]
            ;(prn exchanged)
            {:result true
             :msg    :msg-ok
             :prizes (map transfer prizes)}))
        history-fn
        (fn [{:keys [start-index number]}]
          (when (every? number? [start-index number])
            (let [records (显示兑换历史 兑换历史 start-index number)
                  transfer
                  (fn [{:keys [username time prize]}]
                    {:username username
                     :time     time
                     :prize    prize})]
              ;(prn records)
              {:result       true
               :msg          :msg-ok
               :total-number (显示历史长度 兑换历史)
               :history      (map transfer records)})))
        rank-fn
        (fn [_]
          (let [ranks (显示排行榜 兑换排行 max-rank)
                transfer
                (fn [[user tickets]]
                  {:username user
                   :ticket   tickets})]
            ;(prn ranks)
            {:result true
             :msg    :msg-ok
             :rank   (map transfer ranks)}))
        exchange-fn
        (fn [{:keys [username prize]} time-fn order-id-fn]
          (when (every? string? [username prize])
            (let [time (time-fn)
                  order-id (order-id-fn)
                  record (兑换奖品 兑换服务 username time prize order-id)]
              ;(prn record)
              (case (:result record)
                :ok {:result      true
                     :msg         :msg-exchange-ok
                     :order-id    (:order-id record)
                     :attach-info (attach-info msg record)}
                :not-enough-ticket {:result false
                                    :msg    :msg-exchange-no-ticket}
                :no-more-prize {:result false
                                :msg    :msg-exchange-no-prize}
                {:result false
                 :msg    :msg-svr-err}))))
        post-addr-fn
        (fn [{:keys [order-id recipient addr phone postcode]}]
          (when (every? string? [order-id recipient addr phone postcode])
            (case (保存实物奖品记录 保存服务
                            {:order-id  order-id :address addr :postcode postcode
                             :recipient recipient :phone phone})
              :no-order {:result false
                         :msg    :msg-post-addr-no-order}
              :no-need-addr {:result false
                             :msg    :msg-post-addr-no-need}
              :addr-repeat {:result false
                            :msg    :msg-post-addr-repeat}
              :ok {:result true
                   :msg    :msg-post-addr-ok}
              {:result false
               :msg    :msg-svr-err})))]
    (fn [time-fn order-id-fn {:keys [type] :as req}]
      ;(prn req)
      (-> (case type
            "activity-status" (activity-status-fn req time-fn)
            "user-ticket" (user-ticket-fn req)
            "user-history" (user-history-fn req)
            "prize-num" (prize-num-fn req)
            "history" (history-fn req)
            "rank" (rank-fn req)
            "exchange" (exchange-fn req time-fn order-id-fn)
            "post-addr" (post-addr-fn req)
            nil)
          ((partial merge {:result false :msg :msg-err-cmd}))
          (update :msg msg)))))

(defrecord web服务 [name web-cfg url topic web-params time-nodes prizes max-rank msg
                  奖券查询 兑换状态 兑换服务 保存服务 兑换历史 兑换排行 db]
  component/Lifecycle
  (start [{:keys [stop-fn] :as this}]
    (if stop-fn
      this
      (do
        (println "start" name)
        (let [web-handler (mk-web-handler topic web-params time-nodes prizes max-rank msg
                                          奖券查询 兑换状态 兑换服务 保存服务 兑换历史 兑换排行 db)
              web-handler! (partial web-handler u/now-time u/rand-string)]
          (assoc this :stop-fn (web/start-web-server web-cfg [[url web-handler!]])
                      :web-handler web-handler)))))
  (stop [{:keys [stop-fn] :as this}]
    (if stop-fn
      (do
        (println "stop" name)
        (stop-fn)
        (dissoc this :stop-fn :web-handler))
      this)))

; ==============================================
; 系统
; ==============================================

(defn mk-act
  "构建活动服务"
  [{:keys [db-cfg finance-cfg sso-cfg web-cfg logstash-cfg
           prize-filter daily-reset-time max-rank max-history msg]}
   {:keys [topic games time-nodes tickets prizes file-name web-params]}]
  (let [event-bus (a/chan)
        multi (a/mult event-bus)
        now-time (u/now-time)
        db (dp/mk-exchange-db db-cfg)
        ;配置
        {:keys [host port]} sso-cfg
        {:keys [hour minute]} daily-reset-time
        ticket-thresholds (c/ticket-thresholds (vals tickets))
        ticket-comp-order (c/ticket-comp-order (vals tickets))
        top-ticket-name (:name (first (sort-by :level (vals tickets))))
        ticket-names (keys tickets)
        prize-names (keys prizes)
        day-limit (u/map-vals :day-limit prizes)
        all-limit (u/map-vals :all-limit prizes)
        card-prizes (into {} (filter (fn [[_ v]] (= (:type v) :card)) prizes))
        ;还原所有卡密
        all-cards (u/map-vals #(-> (:card-file %)
                                   (u/read-file-by-line)
                                   (ep/str-cards->map-cards)) card-prizes)
        ;还原已兑换卡密
        card-records (u/map-vals #(dp/prize-records db topic (:name %)) card-prizes)
        provided-cards (u/map-vals #(map (fn [m] (select-keys m [:card :code])) %) card-records)
        ;还原各奖品的兑换数
        {:keys [act-start act-limit]} time-nodes
        zero-exchanged (u/map-vals (constantly 0) prizes)
        all-exchanged (into zero-exchanged (dp/prizes-count db topic [act-start act-limit] prize-names))
        day-exchanged (into zero-exchanged (dp/prizes-count db topic (u/day-time-period now-time) prize-names))
        ;还原排行
        init-rank (dp/users-exchanged-ticket db topic top-ticket-name)
        ;还原历史记录
        init-history (dp/recent-records db topic prize-filter max-history)]
    (component/system-map
      :db db
      :fn-pub (fn [event] (a/put! event-bus event))
      ;奖券相关
      :奖券生成 (-> (map->奖券生成 {:topic topic :time-nodes time-nodes :games games :thresholds ticket-thresholds})
                (component/using [:db]))
      :奖券查询 (-> (map->奖券查询 {:topic topic :ticket-names ticket-names :comp-order ticket-comp-order})
                (component/using [:db]))
      ;奖品相关
      :卡密发放 (mk-卡密发放 all-cards provided-cards)
      :悟性发放 (rp/mk-finance-svc finance-cfg)
      ;兑换相关
      :兑换状态 (ep/mk-exchange-status day-limit all-limit day-exchanged all-exchanged)
      :保存服务 (-> (map->保存服务 {:topic topic :file-name file-name})
                (component/using [:db]))
      :兑换服务 (-> (map->兑换服务 {:topic topic :prizes prizes})
                (component/using [:奖券查询 :兑换状态 :保存服务 :卡密发放 :悟性发放 :fn-pub]))
      ;web服务
      :验证服务 (client/rpc-endpoint :server host :port port)
      :日志服务 (-> (map->logstash服务 {:name "logstash server" :web-cfg logstash-cfg :url "/logstash"})
                (component/using [:奖券生成]))
      :网页服务 (-> (map->web服务
                  {:name   "web server" :web-cfg web-cfg :url "/exchange"
                   :topic  topic :web-params web-params :time-nodes time-nodes
                   :prizes prizes :max-rank max-rank :msg msg})
                (component/using [:奖券查询 :兑换状态 :兑换服务 :保存服务 :兑换历史 :兑换排行 :db]))
      ;其它
      :兑换排行 (mk-兑换排行 (tap-ch multi) init-rank top-ticket-name)
      :兑换历史 (mk-兑换历史 (tap-ch multi) max-history prize-filter init-history)
      :重置服务 (-> (map->重置日兑换状态 {:name "reset server" :hour hour :minute minute})
                (component/using [:兑换状态])))))

(defn mk-system
  "构造系统"
  [config]
  (let [act-cfg-file (:act-cfg config)]
    (component/system-map
      :管理服务 1 #_(-> (map->管理服务
                      {:name    "管理服务" :web-cfg (:web-cfg config) :url "/manage"
                       :sys-cfg config :act-cfg-file act-cfg-file})
                    (component/using [:活动服务]))
      :活动服务 (atom (-> (mk-act config (u/load-edn-config act-cfg-file))
                      (component/start-system))))))

(defn -main [config-file-name & _]
  (-> (mk-system (u/load-edn-config config-file-name))
      (component/start-system)))
