(ns exchange.core
  (:require [clojure.string :as s]
            [exchange.util :as u]))

; ==============================================
; 格式化logstash输入
; ============================================== 

(defn logtime->millis
  "将logstash的时间格式转换为北京时间的时间戳"
  [time-str]
  (-> time-str
      (s/replace " " "T")
      (s/replace "," ".")
      (u/time-str->local-long)))

(defn log->records
  "格式化日志,生成游戏记录"
  [log-map]
  (let [time (logtime->millis (get log-map "time" ""))
        game (get log-map "game")
        users (get log-map "users")]
    (map (fn [u] {:username (get u "username")
                  :win      (get u "win")
                  :game     game
                  :time     time})
         users)))

; ==============================================
; 生成奖券
; ============================================== 

(defn act-open?
  "过滤时间"
  [{:keys [act-start act-limit]} {:keys [time]}]
  {:pre [(number? time)]}
  (and (<= act-start time) (< time act-limit)))

(defn suited-game?
  "过滤游戏"
  [games {:keys [game]}]
  {:pre [(string? game)]}
  (contains? games game))

(defn ticket-thresholds
  "生成奖券阈值结构,元素为包含threshold的map,从小到大排序"
  [tickets]
  (sort-by :threshold tickets))

(defn record->ticket
  "将用户的游戏记录转成用户奖券"
  [topic thresholds {:keys [username win] :as record}]
  {:pre [(string? username) (number? win)]}
  (let [pred #(<= (:threshold %) win)
        ticket-name (->> thresholds (filter pred) last :name)]
    (when ticket-name
      (assoc record :topic topic :name ticket-name))))

(defn log->tickets
  "解析logstash数据
  将游戏日志转换为一组奖券,并保存
  一条日志数据对应一组奖券结果(奖券或nil),结果数与日志中的用户数相同"
  [time-nodes games thresholds topic log]
  (->> log
       (log->records)
       (filter (partial act-open? time-nodes))
       (filter (partial suited-game? games))
       (map (partial record->ticket topic thresholds))))

; ============================================== 
; 奖券合成
; ============================================== 

(defn comp-tickets
  "合成奖券,返回变化量"
  [comp-order tickets]
  (let [tickets-map (apply merge (map (fn [{:keys [name num]}] {name num}) tickets))
        comp-ticket (fn [t-map {:keys [name to rate]}]
                      (let [num (t-map name)]
                        (merge-with + t-map {name (- (rem num rate) num)
                                             to   (quot num rate)})))
        comped-map (reduce comp-ticket tickets-map comp-order)]
    (->> (merge-with (fnil - 0 0) comped-map tickets-map)
         (map (fn [[k v]] {:name k :num v})))))

(defn ticket-comp-order
  "生成合成顺序的结构
  奖券的level越大越先合成
  返回的元素顺序即代表奖券合成的先后顺序"
  [tickets]
  (let [order (sort-by (comp - :level) tickets)
        name (map :name order)
        comp-rate (map :upgrade-rate order)
        comp-from (butlast name)
        comp-to (rest name)]
    (map (fn [name to rate] {:name name :to to :rate rate}) comp-from comp-to comp-rate)))

; ============================================== 
; 活动状态
; ============================================== 

(defn time->status
  "根据时间节点将时间转化为活动状态"
  [{:keys [show-start act-start act-limit show-limit]} now-millis]
  (condp <= now-millis
    show-limit {:status :close :remain -1}
    act-limit {:status :show :remain -1}
    act-start {:status :open :remain (- act-limit now-millis)}
    show-start {:status :foreshow :remain (- act-start now-millis)}
    {:status :close :remain -1}))
