(ns hitokotonushi.ui
  (:use     (hitokotonushi      common entity)
            (clout              core)
            (compojure          response)
            (hiccup             core def element form page util)
            (ring.util          response))
  (:require (clojure            [string  :as string])
            (clojure.data.codec [base64  :as base64])
            (clojure.java       [io      :as io])
            (clojure.tools      [logging :as logging]))
  (:import  (java.net           URI)
            (java.io            ByteArrayInputStream ByteArrayOutputStream)
            (java.text          DateFormat MessageFormat NumberFormat ParseException)
            (java.util          Date Locale)
            (java.util.zip      GZIPInputStream GZIPOutputStream)))

;; Constants.

(def ^:private search-entities-limit
  200)

;; Layouting functions.

(defn- layout
  [title contents]
  (let [locale-string (string/join "-" (remove string/blank? [(.getLanguage *locale*) (.getCountry *locale*)]))]
    (html5
     {:lang locale-string}  ; Hiccup automatically escapes attributes. No need to use 'h'.
     [:head
      [:meta {:http-equiv "X-UA-Compatible" :content "IE=edge"}]
      [:title (h title)]    ; Hiccup escapes only attributes... So, I use 'h'.
      (include-js  "/lib/jquery/jquery.min.js")
      (include-js  "/lib/jquery-ui/jquery-ui.min.js")
      (include-css "/lib/jquery-ui/themes/smoothness/jquery-ui.min.css")
      (include-js  "/lib/moment/moment.min.js")
      (include-js  (format "/lib/moment/locale/%s.js" (.getLanguage *locale*)))  ; Sorry, using country is hard...
      (include-js  "/lib/bootstrap/bootstrap.min.js")
      (include-css "/lib/bootstrap/css/bootstrap.min.css")
      (include-js  "/lib/eonasdan-bootstrap-datetimepicker/bootstrap-datetimepicker.min.js")
      (include-css "/lib/eonasdan-bootstrap-datetimepicker/css/bootstrap-datetimepicker.min.css")
      (include-js  "/lib/datatables/jquery.dataTables.min.js")
      (include-css "/lib/datatables/css/jquery.dataTables_themeroller.css")
      (include-js  "/lib/datatables-plugins/integration/jqueryui/dataTables.jqueryui.min.js")
      (include-js  "/hitokotonushi.js")
      (include-css "/hitokotonushi.css")
      (include-js  (format "/hitokotonushi-%s.js"  locale-string))
      (include-css (format "/hitokotonushi-%s.css" locale-string))]
     [:body
      [:div.grid-container
       (if (not-blank? title)
         [:header.grid-12
          [:h1 (h title)]])
       contents
       [:footer.grid-12
        [:p
         [:small
          "Powered by"
          [:br]
          "&nbsp;&nbsp;Clojure, Apache Derby, clojure.java.jdbc, clojure.data.codec, Compojure, Hiccup, Logging, jQuery, jQuery UI, bootstrap-datetimepicker, DataTables and"
          [:br]
          "&nbsp;&nbsp;一言主（hito-koto-nushi）."]]]]])))

(defn layout-main
  [title main-contents]
  (layout title
          [:article.grid-12 main-contents]))

(defn layout-nav-main
  [title nav-title nav-contents main-title main-contents]
  (layout title
          (list [:nav.grid-3
                 (if (not-blank? nav-title)
                   [:h2 (h nav-title)])
                 nav-contents]
                [:article.grid-9
                 (if (not-blank? main-title)
                   [:h2 (h main-title)])
                 main-contents]
                [:div.clear])))

;; Localizing and formatting functions.

(def ^:dynamic *timezone-offset-millisec*)

(def ^:dynamic *integer-format*)

(def ^:dynamic *decimal-format*)

(def ^:dynamic *currency-format*)

(def ^:dynamic *date-format*)

(def ^:dynamic *timestamp-format*)

(defn gmt-date-time
  [local-date-time]
  (doto (.clone local-date-time)
    (.setTime (+ (.getTime local-date-time) *timezone-offset-millisec*))))

(defn local-date-time
  [gmt-date-time]
  (doto (.clone gmt-date-time)
    (.setTime (- (.getTime gmt-date-time) *timezone-offset-millisec*))))

;; Request.

(def ^:dynamic *request*)

;; URI related functions.

(defn http-get-method-uri
  [uri parameters]
  (letfn [(normalize [key value]
            [(url-encode (name key))
             (if value
               (url-encode (if (keyword? value)
                             (name value)
                             (str value))))])]
    (let [[uri query-string] (string/split uri #"\?")
          query-string       (string/join "&" (->> (merge (->> (string/split (or query-string "") #"&")
                                                               (map #(string/split % #"="))
                                                               (group-by first))
                                                          (->> parameters
                                                               (mapcat (fn [[key value]]
                                                                         (if (coll? value)
                                                                           (map #(normalize key %) value)
                                                                           [(normalize key value)])))
                                                               (group-by first)))
                                                   (remove (fn [[key _]] (= key "")))
                                                   (mapcat (fn [[_ keyvals]]
                                                             (->> keyvals
                                                                  (filter #(not (nil? (second %))))
                                                                  (map (partial string/join "=")))))))]
      (string/join "?" (filter not-blank? [uri query-string])))))

(defn encode-http-parameter-value
  [parameter-value]
  (if (not-blank? parameter-value)
    (letfn [(gzip [bs]
              (with-open [byte-array-output-stream (ByteArrayOutputStream.)]
                (with-open [input-stream  (io/input-stream  (ByteArrayInputStream. bs))
                            output-stream (io/output-stream (GZIPOutputStream. byte-array-output-stream))]
                  (io/copy input-stream output-stream))
                (.toByteArray byte-array-output-stream)))]
      (apply str (map char (base64/encode (gzip (.getBytes parameter-value))))))))

(defn decode-http-parameter-value
  [parameter-value]
  (if (not-blank? parameter-value)
    (letfn [(gunzip [bs]
              (with-open [byte-array-output-stream (ByteArrayOutputStream.)]
                (with-open [input-stream  (io/input-stream  (GZIPInputStream. (ByteArrayInputStream. bs)))
                            output-stream (io/output-stream byte-array-output-stream)]
                  (io/copy input-stream output-stream))
                (.toByteArray byte-array-output-stream)))]
      (apply str (map char (gunzip (base64/decode (.getBytes parameter-value))))))))

(defn now-uri
  [& {:keys [extra-parameters removing-parameter-keys]}]
  (http-get-method-uri (:uri *request*)
                       (merge (->> (cond->> (map (fn [[name value]] [(keyword name) value]) (:query-params *request*))
                                     removing-parameter-keys (remove (fn [[key _]] (contains? removing-parameter-keys key))))
                                   (reduce (fn [result [key value]] (assoc result key value)) {}))
                              extra-parameters)))

(defn list-entities-controller-uri
  [{entity-key :key}]
  (http-get-method-uri (format "/%s" (name entity-key)) nil))

(defn edit-entity-controller-uri
  [entity-or-entity-information & {:keys [editing-start-point? extra-parameters]}]
  (let [entity-name             (or (:entity-name entity-or-entity-information) (name (:key entity-or-entity-information)))
        entity-id-or-create-new (or (:id entity-or-entity-information) "create-new")]
    (http-get-method-uri (format "/%s/%s" entity-name entity-id-or-create-new)
                         (merge {:back-to-uri (encode-http-parameter-value (now-uri)), :editing-start-point? editing-start-point?} extra-parameters))))

(defn select-entity-controller-uri
  [entity-or-entity-information {property-key :key} & {:keys [extra-parameters]}]
  (let [entity-name (or (:entity-name entity-or-entity-information) (name (:key entity-or-entity-information)))
        entity-id   (or (:id entity-or-entity-information) 0)]
    (http-get-method-uri (format "/%s/%s/%s/select-entity" entity-name entity-id (name property-key))
                         (merge {:back-to-uri (encode-http-parameter-value (now-uri))} extra-parameters))))

(defn select-entities-controller-uri
  [entity-or-entity-information {property-key :key} & {:keys [extra-parameters]}]
  (let [entity-name (or (:entity-name entity-or-entity-information) (name (:key entity-or-entity-information)))
        entity-id   (or (:id entity-or-entity-information) 0)]
    (http-get-method-uri (format "/%s/%s/%s/select-entities" entity-name entity-id (name property-key))
                         (merge {:back-to-uri (encode-http-parameter-value (now-uri))} extra-parameters))))

(defn back-to-uri
  []
  (decode-http-parameter-value (get-in *request* [:params :back-to-uri])))

(defn editing-start-point?
  []
  (get-in *request* [:params :editing-start-point?]))

(defn postback?
  []
  (= (get *request* :request-method) :post))

;; Entity option related functions.

(defn localized-plural-label-string
  [entity-information]
  (localized-string (format "%s (plural)" (eval-option entity-information :label))))

(declare control-value-from-property-value)

(defn entity-string
  [{entity-entity-name :entity-name :as entity}]
  (let [property-information (->> (vals (property-informations (keyword entity-entity-name)))
                                  (filter #(eval-option % :representative?))
                                  (first))
        property-value       (get entity (or (:key property-information) :name))]
    (str (control-value-from-property-value property-information property-value)
         (if (:deleted? entity)
           (localized-string " (deleted)")))))

(defn condition-property-informations
  [{entity-key :key}]
  (->> (vals (property-informations entity-key))
       (remove #(or (#{:has-many :virtual} (:type %))
                    (eval-option % :not-search-condition?)))))

(defn list-property-informations
  [{entity-key :key}]
  (->> (vals (property-informations entity-key))
       (filter #(or (= (:key %) :name)
                    (eval-option % :shows-in-list?)
                    (eval-option % :representative?)))))

(defn edit-property-informations
  [{entity-key :key}]
  (->> (vals (property-informations entity-key))
       (remove #(eval-option % :no-edit?))))

;; Form.

(def ^:dynamic *post-form-for?*
  false)

(def ^:dynamic *form-entity-information*
  nil)

(def ^:dynamic *form-entity*
  nil)

(defmacro form-for
  [[method action & [entity-information & [entity]]] & body]
  `(binding [*post-form-for?*          (= ~method :post)
             *form-entity-information* ~entity-information
             *form-entity*             ~entity]
     (form-to [~method ~action]
              (list ~@(map (fn [x#] `(doall-recur ~x#)) body)))))

;; Value parsers.

(defn type-selector
  [{property-type :type property-field-type :field-type}]
  (or property-field-type property-type))

(defmulti ^:private control-value-from-property-value'
  (fn [property-information property-value] (type-selector property-information)))

(defmethod control-value-from-property-value' :belongs-to
  [_ property-value]
  (str (:id property-value)))

(defmethod control-value-from-property-value' :has-many
  [_ property-value]
  (map #(str (:id %)) property-value))

(defmethod control-value-from-property-value' :int
  [_ property-value]
  (if (integer? property-value)
    (.format *integer-format* property-value)
    property-value))

(defmethod control-value-from-property-value' :decimal
  [property-information property-value]
  (if (decimal? property-value)
    (.format (if (eval-option property-information :as-currency?)
               *currency-format*
               *decimal-format*)
             property-value)
    property-value))

(defmethod control-value-from-property-value' :boolean
  [_ property-value]
  (if property-value
    "true"
    "false"))

(defmethod control-value-from-property-value' :date
  [_ property-value]
  (if (instance? Date property-value)
    (.format *date-format* property-value)
    property-value))

(defmethod control-value-from-property-value' :timestamp
  [_ property-value]
  (if (instance? Date property-value)
    (.format *timestamp-format* (local-date-time property-value))
    property-value))

(defmethod control-value-from-property-value' :default
  [_ property-value]
  (str property-value))

(defn control-value-from-property-value
  [property-information property-value]
  (if (not (nil? property-value))
    (control-value-from-property-value' property-information property-value)))

(defmulti ^:private property-value-from-control-value'
  (fn [property-information control-value] (type-selector property-information)))

(defmethod property-value-from-control-value' :belongs-to
  [{property-foreign-entity-key :foreign-entity-key} control-value]
  (get-entity property-foreign-entity-key (Integer. control-value) :get-deleted? true))

(defmethod property-value-from-control-value' :has-many
  [{property-foreign-entity-key :foreign-entity-key} control-value]
  (let [control-value (cond->> control-value
                        (not (coll? control-value)) (vector))]
    (distinct (map #(get-entity property-foreign-entity-key (Integer. %)) control-value))))

(defmethod property-value-from-control-value' :int
  [_ control-value]
  (.parse *integer-format* control-value))

(defmethod property-value-from-control-value' :decimal
  [property-information control-value]
  (BigDecimal. (if (eval-option property-information :as-currency?)
                 (try
                   (.parse *currency-format* control-value)
                   (catch ParseException _
                     (.parse *decimal-format* control-value)))
                 (.parse *decimal-format* control-value))))

(defmethod property-value-from-control-value' :boolean
  [_ control-value]
  (= control-value "true"))

(defmethod property-value-from-control-value' :date
  [_ control-value]
  (.parse *date-format* control-value))

(defmethod property-value-from-control-value' :timestamp
  [_ control-value]
  (gmt-date-time (try
                   (.parse *timestamp-format* control-value)
                   (catch ParseException _
                     (.parse *date-format* control-value)))))

(defmethod property-value-from-control-value' :default
  [_ control-value]
  control-value)

(defn property-value-from-control-value
  [property-information control-value]
  (if (not-empty? control-value)
    (property-value-from-control-value' property-information control-value)))

(defn default-operator-control-value
  [{property-field-type :field-type}]
  (if (#{:string :text} property-field-type)
    "like"
    "="))

(defn column-key
  [{property-key :key property-type :type}]
  (cond->> property-key
    (= property-type :belongs-to) (foreign-key-column-key)))

(defmulti ^:private condition-value-from-control-value'
  (fn [property-information control-value] (type-selector property-information)))

(defmethod condition-value-from-control-value' :belongs-to
  [_ control-value]
  (Integer. control-value))

(defmethod condition-value-from-control-value' :string
  [_ control-value]
  (str "%" control-value "%"))

(defmethod condition-value-from-control-value' :text
  [_ control-value]
  (condition-value-from-control-value' {:field-type :string} control-value))

(defmethod condition-value-from-control-value' :default
  [property-information control-value]
  (property-value-from-control-value property-information control-value))

(defn condition-value-from-control-value
  [property-information control-value]
  (if (not-empty? control-value)
    (condition-value-from-control-value' property-information control-value)))

;; UI controls.

(defn- normalized-control-name
  [name]
  (string/replace name #" " "-"))

(defn- icon-class
  [icon-key]
  (format "ui-icon ui-icon-%s" (name icon-key)))

(defelem margin-control
  []
  [:span
   {:style "margin-left: 10px; margin-right: 10px"}])

(defelem button-control'
  [type name value text title]
  [:button
   {:name name, :type type, :value value, :title title}
   (h text)])

(defn button-control
  [type text & [title class]]
  (let [attrs (or (some->> (cond->> nil
                             (not= type :submit) (cons "as-submit")
                             class               (cons class))
                    (string/join " ")
                    (hash-map :class))
                  {})
        name  (normalized-control-name text)]
    (button-control' attrs type name name (localized-string text) title)))

(defn icon-button-control
  [name value icon-key & [title]]
  (button-control' {:class (format "as-submit %s" (icon-class icon-key))} :button (normalized-control-name name) value "" title))

(defn redirect-control
  [uri text & [title]]
  (if *post-form-for?*
    (button-control' {:class "as-submit"} :button "redirect" uri text title)
    (list (hidden-field uri uri)
          (button-control' {:onclick (format "location.href=document.getElementById(\"%s\").value" (h uri))} :button "redirect" "redirect" text title))))

(defn icon-redirect-control
  [uri icon-key & [title]]
  (let [icon-class (icon-class icon-key)]
    (if *post-form-for?*
      (button-control' {:class (format "as-submit %s" icon-class)} :button "redirect" uri "" title)
      (link-to {:title title} uri [:span {:class icon-class :style "display: inline-block; vertical-align: middle"}]))))

(defn label-control
  [{property-key :key :as property-information}]
  (label property-key (h (capitalize' (localized-label-string property-information)))))

(defn datetimepicker-control
  [{property-key :key property-field-type :field-type} control-value]
  (let [[class datetimepicker-format] (let [datetimepicker-date-format (-> (.toLocalizedPattern *date-format*)
                                                                           (string/replace #"y" "Y")
                                                                           (string/replace #"d" "D"))]
                                        (if (= property-field-type :date)
                                          ["as-date" datetimepicker-date-format]
                                          ["as-timestamp" (format "%s HH:mm:ss" datetimepicker-date-format)]))]
    (text-field {:class class :placeholder (string/upper-case datetimepicker-format) :data-date-format datetimepicker-format}
                property-key control-value)))

(defmulti input-control
  (fn [property-information control-value] (type-selector property-information)))

(declare display-control)

(defmethod input-control :belongs-to
  [{property-key :key property-foreign-entity-key :foreign-entity-key :as property-information} control-value]
  (if-not (eval-option property-information :composition?)
    (let [foreign-entity (property-value-from-control-value property-information control-value)]
      (letfn [(as-drop-down []
                (drop-down property-key
                           (->> (concat (get-entities property-foreign-entity-key)
                                        (if (:deleted? foreign-entity)
                                          [foreign-entity]))
                                (map (fn [foreign-entity] [(h (entity-string foreign-entity)) (str (:id foreign-entity))]))
                                (cons ["" ""]))
                           control-value))
              (as-link []
                (list (if foreign-entity
                        (let [foreign-entity-string (entity-string foreign-entity)]
                          (if-not (:deleted? foreign-entity)
                            (redirect-control (edit-entity-controller-uri foreign-entity) foreign-entity-string)
                            [:span (h foreign-entity-string)]))
                        [:span (h (localized-string "not selected"))])
                      (icon-redirect-control (select-entity-controller-uri (or *form-entity* *form-entity-information*) property-information :extra-parameters {:selected-entity control-value}) :pencil)
                      (hidden-field property-key control-value)))]
        (if (eval-option (get @entity-informations property-foreign-entity-key) :as-drop-down?)
          (as-drop-down)
          (as-link))))
    (display-control property-information control-value)))

(declare list-panel)

(defmethod input-control :has-many
  [{property-foreign-entity-key :foreign-entity-key property-foreign-property-key :foreign-property-key :as property-information} control-value]
  (let [foreign-entity-information   (get @entity-informations property-foreign-entity-key)
        foreign-property-information (-> (property-informations property-foreign-entity-key)
                                         (get property-foreign-property-key))]
    (list-panel foreign-entity-information (property-value-from-control-value property-information control-value)
                :command-panel (if-not (eval-option foreign-property-information :composition?)
                                 (let [select-entities-controller-uri (select-entities-controller-uri *form-entity* property-information
                                                                                                      :extra-parameters (if (eval-option foreign-property-information :optional?)
                                                                                                                          {:selected-entities control-value}))]
                                   (icon-redirect-control select-entities-controller-uri :pencil))
                                 (icon-redirect-control (edit-entity-controller-uri foreign-entity-information :extra-parameters {property-foreign-property-key (str (:id *form-entity*))}) :plus)))))

(defmethod input-control :text
  [{property-key :key} control-value]
  (text-area property-key control-value))  ; Value of <textarea> is not attribute. But hiccup escapes it.

(defmethod input-control :boolean
  [{property-key :key} control-value]
  (drop-down property-key
             (->> (map (fn [value] [(h (localized-string value)) value]) ["true" "false"])
                  (cons ["" ""]))
             control-value))

(defmethod input-control :date
  [property-information control-value]
  (datetimepicker-control property-information control-value))

(defmethod input-control :timestamp
  [property-information control-value]
  (datetimepicker-control property-information control-value))

(defmethod input-control :virtual
  [property-information control-value]
  [:span
   {:style (eval-option property-information :style-for-input)}
   (h control-value)])

(defmethod input-control :default
  [{property-key :key} control-value]
  (text-field property-key control-value))

(defmulti display-control
  (fn [property-information control-value] (type-selector property-information)))

(defmethod display-control :belongs-to
  [property-information control-value]
  [:span
   (h (if-let [foreign-entity (property-value-from-control-value property-information control-value)]
        (entity-string foreign-entity)
        (localized-string "not selected")))])

(defmethod display-control :has-many
  [property-information control-value]
  [:span
   (h (string/join (localized-string ", ")
                   (map entity-string (property-value-from-control-value property-information control-value))))])

(defmethod display-control :int
  [_ control-value]
  [:span
   {:style "display:block; text-align: right"}
   (h control-value)])

(defmethod display-control :decimal
  [_ control-value]
  (display-control {:field-type :int} control-value))

(defmethod display-control :virtual
  [property-information control-value]
  [:span
   {:style (eval-option property-information :style-for-display)}
   (h control-value)])

(defmethod display-control :default
  [_ control-value]
  [:span (h control-value)])

(defn display-error-control
  [errors]
  [:span.error
   (interpose [:br] (map h errors))])

;; UI panels.

(defn operator-control-key
  [property-key]
  (keyword (str (name property-key) "-operator")))

(defn condition-item-panel
  [{property-key :key property-field-type :field-type :as property-information} errors]
  (let [request-params   (:params *request*)
        operator-control (if (#{:int :decimal :date :timestamp} property-field-type)
                           (let [operator-control-key (operator-control-key property-key)]
                             (drop-down operator-control-key
                                        (map (fn [operator-string] [(h operator-string) operator-string]) ["<=" "=" ">="])
                                        (or (get request-params operator-control-key) "="))))]
    [:div.hitokotonushi-input-panel
     (interpose [:br] (concat [(label-control property-information)]
                              (if operator-control
                                [operator-control])
                              [(input-control property-information (get request-params property-key))]
                              (if (not-empty? errors)
                                [(display-error-control errors)])))]))

(defn condition-command-panel
  []
  [:div.hitokotonushi-command-panel
   (button-control :submit "search")])

(defn condition-panel
  [entity-information errors & {:keys [removing-parameter-keys extra-tags]}]
  (form-for [:get (now-uri :removing-parameter-keys removing-parameter-keys) entity-information]
    (concat (let [else-errors (:else errors)]
              (if (not-empty? else-errors)
                [[:div.hitokotonushi-input-panel
                  (display-error-control else-errors)]]))
            (->> (condition-property-informations entity-information)
                 (map (fn [{property-key :key :as property-information}] (condition-item-panel property-information (get errors property-key)))))
            [(condition-command-panel)]
            (if-let [back-to-uri (get-in *request* [:params :back-to-uri])]
              [(hidden-field :back-to-uri back-to-uri)])
            extra-tags)))

(defn list-panel
  [entity-information entities & {:keys [editing-start-point? command-cell-panel-fn command-panel]}]
  (let [list-property-informations (list-property-informations entity-information)]
    (list [:table.as-datatable
           [:thead
            [:tr
             (->> list-property-informations
                  (map (fn [property-information] [:th (h (capitalize' (localized-label-string property-information)))])))
             [:th]]]
           [:tbody
            (->> entities
                 (map (fn [{entity-entity-name :entity-name :as entity}]
                        [:tr
                         (->> list-property-informations
                              (map (fn [{property-key :key :as property-information}]
                                     [:td
                                      (display-control property-information (control-value-from-property-value property-information (get entity property-key)))])))
                         [:td.command
                          (list (if command-cell-panel-fn
                                  (command-cell-panel-fn entity))
                                (if (and (not (eval-option (get @entity-informations (keyword entity-entity-name)) :no-edit-ui-generation?))
                                         (not (:deleted? entity)))
                                  (icon-redirect-control (edit-entity-controller-uri entity :editing-start-point? editing-start-point?) :pencil)))]])))]]
          (or command-panel
              (->> (extended-entity-informations entity-information)
                   (remove #(eval-option % :no-edit-ui-generation?))
                   (sort-by localized-label-string)
                   (map #(icon-redirect-control (edit-entity-controller-uri % :editing-start-point? editing-start-point?) :plus (capitalize' (localized-label-string %)))))))))

(defn edit-item-panel
  [{property-key :key :as property-information} errors]
  [:div.hitokotonushi-input-panel
   (interpose [:br] (concat [(label-control property-information)]
                            [(input-control property-information (control-value-from-property-value property-information (get *form-entity* property-key)))]
                            (if (not-empty? errors)
                              [(display-error-control errors)])))])

(defn edit-command-panel
  []
  (let [can-delete? (and (not (:inserted? *form-entity*))
                         (not (:deleted?  *form-entity*)))]
    [:div.hitokotonushi-command-panel
     (interpose (margin-control) (if (editing-start-point?)
                                   (concat [(button-control :submit "save changes")]
                                           (if can-delete?
                                             [(button-control :button "delete and save changes" nil "as-dangerous")])
                                           [(button-control :button "discard changes")])
                                   (concat [(button-control :submit "change")]
                                           (if can-delete?
                                             [(button-control :button "delete" nil "as-dangerous")])
                                           [(button-control :button "discard change")])))]))

(defn edit-panel
  [entity-information entity errors]
  (form-for [:post (now-uri) entity-information entity]
    (concat (->> (edit-property-informations entity-information)
                 (map (fn [{property-key :key :as property-information}] (edit-item-panel property-information (get errors property-key)))))
            [(edit-command-panel)])))

;; Views.

(defn- list-entities-view
  [entity-information entities errors]
  (layout-nav-main (capitalize' (format (localized-string "list of entities")
                                        (localized-plural-label-string entity-information)))
                   (localized-string "search conditions")
                   (condition-panel  entity-information errors)
                   (localized-string "search results")
                   (list (list-panel entity-information entities :editing-start-point? "yes")
                         (form-for [:post (now-uri)]
                           [:div.hitokotonushi-command-panel
                            (button-control :button "back to top page")]))))

(defn- edit-entity-view
  [entity-information entity errors]
  (layout-main (capitalize' (format (localized-string "edit entity")
                                    (localized-label-string entity-information)))
               (edit-panel entity-information entity errors)))

(defn- select-entity-view
  [entity-information {property-foreign-entity-key :foreign-entity-key :as property-information} selected-entity entities errors]
  (let [foreign-entity-information   (get @entity-informations property-foreign-entity-key)
        selected-entity-hidden-field (hidden-field :selected-entity (control-value-from-property-value property-information selected-entity))]
    (layout-nav-main (capitalize' (format (localized-string "select entity")
                                          (localized-label-string entity-information)
                                          (localized-label-string property-information)))
                     (localized-string "search conditions")
                     (condition-panel  foreign-entity-information errors :removing-parameter-keys #{:selected-entity} :extra-tags (list selected-entity-hidden-field))
                     (localized-string "search results")
                     (form-for [:post (now-uri)]
                       [:div.hitokotonushi-input-panel
                        (list-panel foreign-entity-information entities
                                    :command-cell-panel-fn #(icon-button-control "select entity"   (:id %) :copy))]
                       [:h2 (h (localized-string "selected"))]
                       [:div.hitokotonushi-input-panel
                        (list-panel foreign-entity-information
                                    (if selected-entity
                                      [selected-entity])
                                    :command-cell-panel-fn #(icon-button-control "unselect entity" (:id %) :scissors)
                                    :command-panel         "")]
                       [:div.hitokotonushi-command-panel
                        (button-control :submit "select")
                        (margin-control)
                        (button-control :button "discard select")]
                       selected-entity-hidden-field))))

(defn- select-entities-view
  [entity-information {property-foreign-entity-key :foreign-entity-key :as property-information} selected-entities optional? entities errors]
  (let [foreign-entity-information      (get @entity-informations property-foreign-entity-key)
        selected-entities-hidden-fields (map #(hidden-field :selected-entities %) (control-value-from-property-value property-information selected-entities))]
    (layout-nav-main (capitalize' (format (localized-string "select entity")
                                          (localized-label-string entity-information)
                                          (localized-label-string property-information)))
                     (localized-string "search conditions")
                     (condition-panel  foreign-entity-information errors :removing-parameter-keys #{:selected-entities} :extra-tags selected-entities-hidden-fields)
                     (localized-string "search results")
                     (form-for [:post (now-uri)]
                       [:div.hitokotonushi-input-panel
                        (list-panel foreign-entity-information entities
                                    :command-cell-panel-fn (if optional?
                                                             #(icon-button-control "selected entities" (:id %) :copy)))]
                       [:h2 (h (localized-string "selected"))]
                       [:div.hitokotonushi-input-panel
                        (list-panel foreign-entity-information selected-entities
                                    :command-cell-panel-fn (if optional?
                                                             #(icon-button-control "unselect entity"   (:id %) :scissors))
                                    :command-panel         "")]
                       [:div.hitokotonushi-command-panel
                        (if optional?
                          (list (button-control :submit "select")
                                (margin-control)
                                (button-control :button "discard select"))
                          (button-control :submit "back"))]
                       selected-entities-hidden-fields))))

(defn- error-view
  [message]
  (layout-main (localized-string "error")
               (list [:p (string/replace (h message) #"\n" "<br>")]
                     [:p (link-to "javascript:history.back()" (localized-string "back"))])))

;; Session related functions.

(defn store-session
  [response & key-values]
  (let [response (cond-> response
                   (not (response? response)) (-> (ring.util.response/response)
                                                  (header "Content-Type" "text/html; charset=UTF-8")))]
    (assoc response :session (apply assoc (:session *request*) key-values))))

(defn- editing-entities-session
  []
  (get-in *request* [:session :editing-entities]))

(defn- store-editing-entities-session!
  [response entities]
  (let [entities (cond->> entities
                   (not (coll? entities)) (vector))]
    (store-session response :editing-entities (distinct (concat entities (editing-entities-session))))))

(defn- delete-editing-entities-session!
  [response & [entity]]
  (let [editing-entities (if entity
                           (remove #(= % entity) (editing-entities-session)))]
    (store-session response :editing-entities editing-entities)))

(defn- restore-editing-entities-session!
  []
  (doseq [entity (editing-entities-session)]
    (add-to-cached-entities! entity)))

;; Controllers.

(defn back
  [& {:keys [extra-parameters]}]
  (redirect (or (http-get-method-uri (back-to-uri) extra-parameters) "/")))

(defn- search-entities
  [{entity-key :key :as entity-information} request-params]
  (letfn [(join-where-items' [result [item & more]]
            (if item
              (recur (list 'and item result) more)
              result))
          (join-where-items [[item & more]]
            (join-where-items' item more))]
    (let [errors (atom {})
          where  (->> (condition-property-informations entity-information)
                      (mapcat (fn [{property-key :key property-field-type :field-type :as property-information}]
                                (let [condition-control-value (get request-params property-key)]
                                  (if (not-blank? condition-control-value)
                                    (try
                                      [(list (symbol (or (get request-params (operator-control-key property-key))
                                                         (default-operator-control-value property-information)))
                                             (column-key property-information)
                                             (condition-value-from-control-value property-information condition-control-value))]
                                      (catch ParseException _
                                        (swap! errors assoc property-key [(format (localized-string "item's format is wrong")
                                                                                  (localized-label-string property-information))])))))))
                      (join-where-items))
          entities (if (empty? @errors)
                     (let [entities (eval `(get-entities ~entity-key ~where))]
                       (if (> (count entities) search-entities-limit)
                         (do (swap! errors assoc :else [(localized-string "so much entities are gotten")])
                             nil)
                         entities)))]
      [entities @errors])))

(defn- list-entities-controller
  [entity-information]
  (restore-editing-entities-session!)
  (let [request-params (:params *request*)]
    (cond
      (:back-to-top-page request-params) (redirect "/")
      (:search           request-params) (apply list-entities-view entity-information (search-entities entity-information request-params))
      :else                              (list-entities-view entity-information nil nil))))

(defmacro ^:private do-when-every-entities-are-valid
  [& body]
  `(let [errors# (apply merge-with concat (->> (map vals (vals @*cached-entities*))  ; Validate once more for the pushing [back] button case.
                                               (flatten)
                                               (filter #(and (or (:inserted? %) (:updated? %)) (not (:deleted? %))))
                                               (map validate-entity)))]
     (if (empty? errors#)
       (do ~@body)
       (error-view (format (localized-string "some entities are not validated")
                           (->> errors#
                                (reduce (fn [result# [key# messages#]] (assoc result# key# (string/join "\n" messages#))) {})
                                (reduce (fn [result# [key# message#]] (string/join "\n" [result# (name key#) message#])) "")))))))

(defn- edit-entity-controller
  [{entity-key :key :as entity-information}]
  (restore-editing-entities-session!)
  (letfn [(set-properties-from-request! [entity params]
            (let [errors (atom {})]
              (doseq [{property-key :key :as property-information} (vals (property-informations (keyword (:entity-name entity))))]
                (if-let [control-value (get params property-key)]
                  (try
                    (set-property! entity property-key (property-value-from-control-value property-information control-value))
                    (catch ParseException _
                      (set-property! entity property-key control-value)
                      (swap! errors assoc property-key [(format (localized-string "item's format is wrong")
                                                                (localized-label-string property-information))])))))
              (if (and (empty? @errors) (postback?))
                (validate-entity entity)
                @errors)))]
    (let [request-params (:params *request*)
          entity         (let [id-string-or-create-new (:id-string-or-create-new request-params)]
                           (if (= id-string-or-create-new "create-new")
                             (create-entity! entity-key {})
                             (get-entity entity-key (Integer. id-string-or-create-new) :get-deleted? true)))
          errors         (set-properties-from-request! entity request-params)]
      (cond
        (and (empty? errors) (:save-changes request-params)) (do-when-every-entities-are-valid (save!)
                                                                                               (-> (back)
                                                                                                   (delete-editing-entities-session!)))
        (and (empty? errors) (:change       request-params)) (back)
        (:delete-and-save-changes request-params)            (do-when-every-entities-are-valid (delete-entity! entity)
                                                                                               (save!)
                                                                                               (-> (back)
                                                                                                   (delete-editing-entities-session!)))
        (:delete                  request-params)            (let [deleted-entities (delete-entity! entity)]
                                                               (-> (back)
                                                                   (store-editing-entities-session! deleted-entities)))
        (:discard-changes         request-params)            (-> (back)
                                                                 (delete-editing-entities-session!))
        (:discard-change          request-params)            (-> (back)
                                                                 (delete-editing-entities-session! entity))
        (:redirect                request-params)            (redirect (:redirect request-params))
        :else                                                (binding [*request* (assoc *request* :uri (string/replace (:uri *request*) "create-new" (str (:id entity))))]
                                                               (-> (edit-entity-view entity-information entity errors)
                                                                   (store-editing-entities-session! entity)))))))

(defn- select-entity-controller
  [{entity-key :key :as entity-information} {property-key :key property-type :type property-foreign-entity-key :foreign-entity-key :as property-information}]
  (restore-editing-entities-session!)
  (let [request-params             (:params *request*)
        foreign-entity-information (get @entity-informations property-foreign-entity-key)
        selected-entity            (if (not (:unselect-entity request-params))
                                     (or (property-value-from-control-value property-information (:select-entity   request-params))
                                         (property-value-from-control-value property-information (:selected-entity request-params))))
        [entities errors]          (if (:search request-params)
                                     (search-entities foreign-entity-information request-params))]
    (cond
      (:select         request-params) (back :extra-parameters {property-key (or (control-value-from-property-value property-information selected-entity) "")})
      (:discard-select request-params) (back)
      (:redirect       request-params) (redirect (http-get-method-uri (:redirect request-params)
                                                                      {:back-to-uri (encode-http-parameter-value (now-uri :extra-parameters {:selected-entity (control-value-from-property-value property-information selected-entity)}))}))
      :else                            (select-entity-view entity-information property-information selected-entity entities errors))))

(defn- select-entities-controller
  [{entity-key :key :as entity-information} {property-key :key property-foreign-entity-key :foreign-entity-key property-foreign-property-key :foreign-property-key :as property-information}]
  (restore-editing-entities-session!)
  (let [request-params             (:params *request*)
        foreign-entity-information (get @entity-informations property-foreign-entity-key)
        optional?                  (-> (property-informations (:key foreign-entity-information))
                                       (get property-foreign-property-key)
                                       (eval-option :optional?))
        selected-entities          (if optional?
                                     (->> (cond->> (property-value-from-control-value property-information (:selected-entities request-params))
                                            (:unselect-entity request-params) (remove #(= (str (:id %)) (:unselect-entity request-params))))
                                          (distinct))
                                     (get (get-entity entity-key (Integer. (:id-string request-params)) :get-deleted? true) property-key))
        [entities errors]          (if (:search request-params)
                                     (search-entities foreign-entity-information request-params))]
    (cond
      (:select         request-params) (-> (back :extra-parameters {property-key (or (control-value-from-property-value property-information selected-entities) "")})
                                           (store-editing-entities-session! selected-entities))
      (:discard-select request-params) (back)
      (:back           request-params) (back)
      (:redirect       request-params) (redirect (if optional?
                                                   (http-get-method-uri (:redirect request-params)
                                                                        {:back-to-uri (encode-http-parameter-value (now-uri :extra-parameters {:selected-entities (control-value-from-property-value property-information selected-entities)}))})
                                                   (:redirect request-params)))
      :else                            (-> (select-entities-view entity-information property-information selected-entities optional? entities errors)
                                           (store-editing-entities-session! selected-entities)))))

;; Request processors.

(defn- process-set-timezone-offset-request
  [request]
  (if (route-matches "/set-timezone-offset" request)
    (render (html5
             [:body
              (hidden-field :go-back-to-uri (.getPath (URI. (get-in request [:params :go-back-to-uri]))))  ; Using hidden tag, because generating javascript code from parameters is so dangerous.
              (javascript-tag "document.cookie = \"timezone-offset=\" + new Date().getTimezoneOffset(); location.href = document.getElementById(\"go-back-to-uri\").value;")])
            request)))

(defn- process-hitokotonushi-ui-request
  [request {entity-key :key :as entity-information}]
  (if-let [response (letfn [(merge-request-params [params] (merge-with merge request {:params params}))]
                      (let [entity-name (name entity-key)]
                        (condp route-matches request
                          (format "/%s"                                           entity-name) :>> (fn [_]
                                                                                                     (when-not (eval-option entity-information :no-ui-generation?)
                                                                                                       (list-entities-controller request entity-information)))
                          (format "/%s/:id-string-or-create-new"                  entity-name) :>> (fn [params]
                                                                                                     (when-not (eval-option entity-information :no-edit-ui-generation?)
                                                                                                       (edit-entity-controller (merge-request-params params) entity-information)))
                          (format "/%s/:id-string/:property-name/select-entity"   entity-name) :>> (fn [{property-name :property-name :as params}]
                                                                                                     (select-entity-controller   (merge-request-params params) entity-information (get (property-informations entity-key) (keyword property-name))))
                          (format "/%s/:id-string/:property-name/select-entities" entity-name) :>> (fn [{property-name :property-name :as params}]
                                                                                                     (select-entities-controller (merge-request-params params) entity-information (get (property-informations entity-key) (keyword property-name))))
                          nil)))]
    (render response request)))

(defn hitokotonushi-ui
  []
  (fn [request]
    (or (process-set-timezone-offset-request request)
        (some #(process-hitokotonushi-ui-request request %) (vals @entity-informations)))))

;; Aspect.

(defn controller-aspect
  [symbol function [request :as args]]
  (try
    (if-let [timezone-offset-minute-string (get-in request [:cookies "timezone-offset" :value])]
      (binding [*timezone-offset-millisec* (* (Integer/parseInt timezone-offset-minute-string) 60 1000)
                *locale*                   (letfn [(localizing-javascript-exists? [culture]
                                                     (.getResource (.getContextClassLoader (Thread/currentThread)) (format "public/hitokotonushi-%s.js" culture)))]
                                             (let [[_ language country] (map str (re-find #"^(\w+)(?:-(\w+))?" (or (get-in request [:headers "accept-language"]) "")))
                                                   [  language country] (or (some #(if (localizing-javascript-exists? (first %))
                                                                                     (next %))
                                                                                  (concat (if (not-blank? country)
                                                                                            [[(format "%s-%s" language country) language country]])
                                                                                          [[language language ""]]))
                                                                            ["en" "US"])]
                                               (Locale. language country)))]
        (binding [*messages*         (load-messages)
                  *integer-format*   (NumberFormat/getIntegerInstance *locale*)
                  *decimal-format*   (doto (NumberFormat/getNumberInstance *locale*)
                                       (.setMaximumFractionDigits decimal-scale)
                                       (.setMinimumFractionDigits decimal-scale))
                  *currency-format*  (NumberFormat/getCurrencyInstance (Locale. (.getLanguage *locale*) country))
                  *date-format*      (DateFormat/getDateInstance DateFormat/MEDIUM *locale*)
                  *timestamp-format* (DateFormat/getDateTimeInstance DateFormat/MEDIUM DateFormat/MEDIUM *locale*)]
          (letfn [(normalize-params [params]
                    (->> params
                         (map (fn [[key value]] [(keyword key) value]))
                         (array-map')))]
            (binding [*request* (assoc request :params (merge (:params request) (normalize-params (:query-params request)) (normalize-params (:form-params request))))]
              (hitokotonushi-session nil
                                     (apply function (next args)))))))  ; remove first parameter "request".
      (redirect (format "/set-timezone-offset?go-back-to-uri=%s" (url-encode (http-get-method-uri (:uri request) (:query-params request))))))
    (catch Exception ex
      (logging/error ex "an error is occured")
      (error-view (.getMessage ex)))))

(weave-aspect 'hitokotonushi.ui #".*-controller" controller-aspect)
