(ns hitokotonushi.entity
  (:use     (hitokotonushi common))
  (:require (clojure       [string  :as string])
            (clojure.java  [jdbc    :as jdbc])
            (clojure.tools [logging :as logging]))
  (:import  (clojure.lang  ILookup)
            (java.sql      SQLException Timestamp)
            (java.util     Date)))

(weave-aspect 'clojure.java.jdbc #"query|execute!"
  (fn [symbol function args]
    (logging/infof "SQL: %s" (vec (second args)))
    (apply function args)))

(def string-max-length
  256)

(def text-max-length
  4096)

(def decimal-scale
  10)

(def country
  "JP")  ; Please change. This will be used when showing currency.

(def ^:private column-types
  {:string    ["VARCHAR" (format "(%d)" string-max-length)]
   :text      ["VARCHAR" (format "(%d)" text-max-length)]
   :int       ["INTEGER"]
   :decimal   ["DECIMAL" (format "(30, %d)" decimal-scale)]
   :boolean   ["BOOLEAN"]
   :date      ["DATE"]
   :timestamp ["TIMESTAMP"]})

(def ^:private no-parameter-entity-option-keys
  #{:no-ui-generation :no-edit-ui-generation :as-drop-down})

(def ^:private no-parameter-property-option-keys
  #{:optional :composition :representative :not-search-condition :shows-in-list :no-edit :as-currency})

(def ^:private no-parameter-option-keys
  (set (concat no-parameter-entity-option-keys no-parameter-property-option-keys)))

(def ^:private entity-information-requests
  (atom []))

(defn defentity'
  [key & spec]
  (letfn [(property-informations [result [[property-key & property-spec] & more]]
            (letfn [(property-information []
                      (let [[[property-type foreign-entity-key _ foreign-property-key] _ optional-property-specs] (partition-by #(= % :options) property-spec)
                            property-base    (letfn [(has-many-property-base []
                                                       {:type                 :has-many
                                                        :foreign-entity-key   foreign-entity-key
                                                        :foreign-property-key (or foreign-property-key key)})
                                                     (belongs-to-property-base []
                                                       {:type                 :belongs-to
                                                        :foreign-entity-key   (or foreign-entity-key property-key)})
                                                     (virtual-property-base []
                                                       {:type                 :virtual})]
                                               (case property-type
                                                 :has-many   (has-many-property-base)
                                                 :belongs-to (belongs-to-property-base)
                                                 :virtual    (virtual-property-base)
                                                 {:type       :field
                                                  :field-type property-type}))
                            property-options (let [options-base (apply array-map optional-property-specs)]
                                               (cond-> options-base
                                                 (not (:label options-base)) (assoc :label (str (name key) "." (name property-key)))
                                                 (not (:<=    options-base)) (cond-> (= property-type :string) (assoc :<= string-max-length)
                                                                                     (= property-type :text)   (assoc :<= text-max-length))))]
                        (-> property-base
                            (assoc :options property-options
                                   :key     property-key))))]
              (if property-key
                (recur (assoc result property-key (property-information)) more)
                result)))]
    (let [[[_ base-entity-key & property-specs] _ optional-specs] (->> (cond->> spec
                                                                         (not= (first spec) :extends) (concat [:extends nil]))
                                                                       (partition-by #(= % :options)))
          options (let [options-base (apply array-map optional-specs)]
                    (cond-> options-base
                      (not (:label options-base)) (assoc :label (str (name key)))))]
      (swap! entity-information-requests conj {:key                   key
                                               :base-entity-key       base-entity-key
                                               :property-informations (property-informations {} (reverse property-specs))  ; array-mapは追加された要素が先頭に挿入される（順序が逆になる）ので、あらかじめ逆順にしておきます。
                                               :options               options}))))

(defmacro defentity
  [key & spec]
  (letfn [(normalize-option-specs [result [key & [parameter & more :as more-when-no-parameter-option-key]]]
            (if key
              (let [key (keyword key)]
                (if (no-parameter-option-keys key)
                  (recur (concat [(keyword (str (name key) "?")) true]      result) more-when-no-parameter-option-key)
                  (recur (concat [key                            parameter] result) more)))
              result))
          (normalize-spec [x]
            (condp' x
              coll?   (let [[base-specs separators option-specs] (partition-by #(= % :options) x)]
                        (vec (concat (map normalize-spec base-specs) separators (normalize-option-specs [] option-specs))))
              symbol? (keyword x)
              x))]
    (let [key  (keyword key)
          spec (normalize-spec spec)]
      `(do (logging/info "defentity" ~key ~@spec)
           (defentity' ~key ~@spec)))))

(def entity-informations
  (atom {}))

(defn foreign-key-column-key
  [property-key]
  (keyword (str (name property-key) "-id")))

(defn extended-entity-informations
  [entity-information]
  (letfn [(directly-extended-entity-informations [{key :key}]
            (->> @entity-informations
                 (vals)
                 (filter #(= (:base-entity-key %) key))))]
    (tree-seq map? directly-extended-entity-informations entity-information)))

(defn extended-entity-keys
  [entity-key]
  (->> (get @entity-informations entity-key)
       (extended-entity-informations)
       (map :key)))

(def db-spec
  (atom nil))

(defn hitokotonushi-init
  [db-spec]
  (letfn [(reset-db-spec []
            (reset! hitokotonushi.entity/db-spec db-spec))
          (assoc-entity-information-requests []
            (doseq [{key :key :as entity-information} @entity-information-requests]
              (swap! entity-informations assoc key entity-information)))
          (create-or-validate-tables[]
            (doseq [[key columns] (->> @entity-information-requests
                                       (mapcat #(if-not (:base-entity-key %)
                                                  (let [extended-entity-informations (extended-entity-informations %)
                                                        sti?                         (> (count extended-entity-informations) 1)]
                                                    [[(:key %)
                                                      (cond->> (->> extended-entity-informations
                                                                    (map :property-informations)
                                                                    (apply merge)
                                                                    (vals)
                                                                    (mapcat (fn [{type :type :as property-information}]
                                                                              (cond
                                                                                (= type :field)      [(cons (:key property-information) (get column-types (:field-type property-information)))]
                                                                                (= type :belongs-to) [[(foreign-key-column-key (:key property-information)) "INTEGER"]])))
                                                                    (concat [[:id          "INTEGER" "NOT NULL PRIMARY KEY"]
                                                                             [:modified-at "TIMESTAMP"]
                                                                             [:deleted?    "BOOLEAN"]]))
                                                        sti? (concat [[:entity-name "VARCHAR" "(256)"]]))]]))))]
              (letfn [(table-exists? []
                        (> (->> (db-metadata db-spec #(.getTables % nil nil (name key) (into-array ["TABLE"])))
                                (count))
                           0))
                      (create-table []
                        (jdbc/execute! db-spec
                          [(format "CREATE TABLE %s (%s)"
                                   (as-quoted-sql-name key)
                                   (->> columns
                                        (map (fn [[column-key & column-spec]]
                                               (string/join " " (cons (as-quoted-sql-name column-key) column-spec))))
                                        (string/join ", ")))]
                          :transaction? false))
                      (validate-table []
                        (let [actual-columns  (->> (db-metadata db-spec #(.getColumns % nil nil (name key) nil))
                                                   (map (fn [{column-name :column_name type-name :type_name}]
                                                          [(keyword column-name) type-name])))
                              correct-columns (->> columns
                                                   (map (partial take 2)))]
                          (when-not (= actual-columns correct-columns)
                            (throw (IllegalStateException. (localized-string "database schema is not matched with entities"))))))]
                (if-not (table-exists?)
                  (create-table)
                  (validate-table)))))
          (create-sequence-if-needed []
            (try
              (jdbc/query db-spec [(format "VALUES (NEXT VALUE FOR %s)" (as-quoted-sql-name :id-sequence))])
              (catch SQLException _
                (jdbc/execute! db-spec [(format "CREATE SEQUENCE %s START WITH 0" (as-quoted-sql-name :id-sequence))]))))
          (reset-entity-information-requests []
            (reset! entity-information-requests []))]
    (try
      (reset-db-spec)
      (assoc-entity-information-requests)
      (create-or-validate-tables)
      (create-sequence-if-needed)
      (finally
        (reset-entity-information-requests)))))

(def ^:dynamic *db*
  nil)

(def ^:dynamic *cached-entities*
  nil)

(defmacro hitokotonushi-session
  [& body]
  `(jdbc/with-db-transaction [db# @db-spec]
     (binding [*cached-entities* (atom {})
               *db*              db#]
       ~@body)))

(declare create-entity!)

(defn- base-entity-informations
  [entity-information]
  (if entity-information
    (cons entity-information (lazy-seq (base-entity-informations (get @entity-informations (:base-entity-key entity-information)))))))

(defn root-entity-key
  [entity-key]
  (:key (last (base-entity-informations (get @entity-informations entity-key)))))

(defn sti-where-sql
  [entity-key]
  (format " AND (%s.%s IN (%s))"
          (as-quoted-sql-name entity-key)
          (as-quoted-sql-name :entity-name)
          (string/join ", " (map (jdbc/as-sql-name (jdbc/quoted \')) (extended-entity-keys entity-key)))))

(defn sti-pred
  [entity-key]
  #((set (extended-entity-keys entity-key)) (keyword (:entity-name %))))

(defn get-entity
  [entity-key id & {:keys [get-deleted?]}]
  (let [root-entity-key (root-entity-key entity-key)
        sti?            (not= root-entity-key entity-key)]
    (letfn [(get-entity-from-cached-entities []
              (if-let [entity (get-in @*cached-entities* [root-entity-key id])]
                (if (or (not sti?)
                        ((sti-pred entity-key) entity))
                  entity)))
            (get-entity-from-database []
              (if-let [record (first (jdbc/query *db*
                                       [(cond-> (format "SELECT %1$s.* FROM %2$s AS %1$s WHERE (%1$s.%3$s = ?)"
                                                        (as-quoted-sql-name entity-key)
                                                        (as-quoted-sql-name root-entity-key)
                                                        (as-quoted-sql-name :id))
                                          (not get-deleted?) (str (format " AND (NOT %s.%s)"
                                                                          (as-quoted-sql-name entity-key)
                                                                          (as-quoted-sql-name :deleted?)))
                                          sti?               (str (sti-where-sql entity-key)))
                                        id]))]
                (create-entity! entity-key record)))]
      (if-let [entity (get-entity-from-cached-entities)]
        (if (or (not (:deleted? entity))
                get-deleted?)
          entity)
        (get-entity-from-database)))))

(defn fill-cached-entities
  [entity-key sql-parameters]
  (doall (->> (jdbc/query *db* sql-parameters)
              (map #(or (get-in @*cached-entities* [(root-entity-key entity-key) (:id %)])
                        (create-entity! entity-key %))))))

(defn get-entities-from-cached-entities
  [entity-key pred]
  (->> (get @*cached-entities* (root-entity-key entity-key))
       (vals)
       (filter pred)))

(defn join-sql-items
  [x entity-key]
  (condp' x
    keyword? (letfn [(join-sql-items' [result entity-information [entity-key & [property-key & more-property-keys :as more]]]
                       (if more-property-keys
                         (let [property-information (get (:property-informations entity-information) property-key)
                               foreign-entity-key   (:foreign-entity-key property-information)]
                           (recur (conj result
                                        (format "JOIN %1$s AS %2$s ON %2$s.%3$s = %4$s.%5$s"
                                                (as-quoted-sql-name foreign-entity-key)
                                                (as-quoted-sql-name property-key)
                                                (as-quoted-sql-name :id)
                                                (as-quoted-sql-name entity-key)
                                                (as-quoted-sql-name (foreign-key-column-key property-key))))
                                  (get @entity-informations foreign-entity-key)
                                  more))
                         result))]
               (join-sql-items' [] (get @entity-informations entity-key) (cons entity-key (map keyword (string/split (name x) #"\.")))))
    nil))

(defn where-sql-item
  [x y entity-key]
  (condp' x
    keyword? (let [[property-key entity-alias-key] (->> (string/split (name x) #"\.")
                                                        (reverse)
                                                        (take 2)
                                                        (map keyword))]
               (format "%s.%s"
                       (as-quoted-sql-name (or entity-alias-key entity-key))
                       (as-quoted-sql-name (cond->> property-key
                                             (instance? ILookup y) (foreign-key-column-key)))))
    nil?     "null"
    "?"))

(defn where-parameter
  [x]
  (condp' x
    keyword?                    nil
    nil?                        nil
    (partial instance? ILookup) (:id x)
    x))

(defn pred-item
  [x entity]
  (condp' x
    keyword? (letfn [(property-value [entity [property-name & more-property-names]]
                       (let [value (get entity (keyword property-name))]
                         (if more-property-names
                           (recur value more-property-names)
                           value)))]
               (-> (property-value entity (string/split (name x) #"\."))
                   (recur entity)))
    (partial instance? ILookup) (:id x)
    x))

(defmacro get-entities
  ([entity-key sql-parameters pred]
     (if pred
       `(do (fill-cached-entities ~entity-key ~sql-parameters)
            (->> (get-entities-from-cached-entities ~entity-key ~pred)
                 (sort-by :id)))
       `(->> (fill-cached-entities ~entity-key ~sql-parameters)
             (sort-by :id))))
  ([entity-key join-sql where-sql where-parameters pred]
     `(let [root-entity-key# (root-entity-key ~entity-key)
            sti?#            (not= root-entity-key# ~entity-key)]
        (get-entities ~entity-key
                      (cons (cond-> (format "SELECT %1$s.* FROM %2$s AS %1$s %3$s WHERE %4$s"
                                            (as-quoted-sql-name ~entity-key)
                                            (as-quoted-sql-name root-entity-key#)
                                            ~join-sql
                                            ~where-sql)
                              sti?# (str (sti-where-sql ~entity-key)))
                            ~where-parameters)
                      (cond-> ~pred
                        sti?# (every-pred ~pred (sti-pred ~entity-key))))))
  ([entity-key]
     `(get-entities ~entity-key
                    ""
                    (format "(%s.%s = FALSE)" (as-quoted-sql-name ~entity-key) (as-quoted-sql-name :deleted?))
                    []
                    #(not (:deleted? %))))
  ([entity-key condition]
     (if-not (empty? condition)
       (let [condition# (list 'and '(= :deleted? false) condition)]
         (letfn [(join-sqls [[op# x# y#]]
                   (condp contains? op#
                     #{'not}                           [(join-sqls x#)]
                     #{'and 'or}                       [(join-sqls x#) (join-sqls y#)]
                     #{'= 'is '<> '< '<= '> '>= 'like} [`(join-sql-items ~x# ~entity-key) `(join-sql-items ~y# ~entity-key)]
                     #{'in}                            [`(join-sql-items ~x# ~entity-key)]))
                 (where-sql [[op# x# y#]]
                   (let [str-op# (string/upper-case op#)]
                     `(format "(%s)"
                              (string/join " "
                                           ~(condp contains? op#
                                              #{'not}                           [str-op# (where-sql x#)]
                                              #{'and 'or}                       [(where-sql x#) str-op# (where-sql y#)]
                                              #{'= 'is '<> '< '<= '> '>= 'like} [`(where-sql-item ~x# ~y# ~entity-key) str-op# `(where-sql-item ~y# ~x# ~entity-key)]
                                              #{'in}                            [`(format "%s.%s" (as-quoted-sql-name ~entity-key) (as-quoted-sql-name ~x#)) str-op# `(str "(" (string/join ", " (repeat (count ~y#) "?")) ")")])))))
                 (where-parameters [[op# x# y#]]
                   (condp contains? op#
                     #{'not}                           [(where-parameters x#)]
                     #{'and 'or}                       [(where-parameters x#) (where-parameters y#)]
                     #{'= 'is '<> '< '<= '> '>= 'like} [`(where-parameter ~x#) `(where-parameter ~y#)]
                     #{'in}                            [y#]))  ; (take 10 (iterate inc 10))のようなコードの場合に分解されないよう、ベクター化します。
                 (pred [[op# x# y#]]
                   (condp contains? op#
                     #{'not}             `(~op# ~(pred x#))
                     #{'and 'or}         `(~op# ~(pred x#) ~(pred y#))
                     #{'= '< '<= '> '>=} `(if (or (= (class ~x#) java.util.Date)
                                                  (= (class ~y#) java.util.Date))
                                            (~op# (.getTime (pred-item ~x# ~'-entity-)) (.getTime (pred-item ~y# ~'-entity-)))
                                            (~op# (pred-item ~x# ~'-entity-) (pred-item ~y# ~'-entity-)))
                     #{'is}              `(=    (pred-item ~x# ~'-entity-) (pred-item ~y# ~'-entity-))
                     #{'<>}              `(not= (pred-item ~x# ~'-entity-) (pred-item ~y# ~'-entity-))
                     #{'like}            (let [item-x# `(pred-item ~x# ~'-entity-)]
                                           `(if (= (first ~y#) \%)
                                              (if (= (last ~y#) \%)
                                                (>= (.indexOf ~item-x# (apply str (butlast (next ~y#)))) 0)
                                                (.endsWith ~item-x# (apply str (next ~y#))))
                                              (.startsWith ~item-x# (apply str (butlast ~y#)))))
                     #{'in}              `((set ~y#) (pred-item ~x# ~'-entity-))))]
           `(get-entities ~entity-key
                          (string/join " " (distinct (remove nil? (flatten ~(join-sqls condition#)))))
                          ~(where-sql condition#)
                          (remove nil? (flatten ~(where-parameters condition#)))
                          (fn [~'-entity-] ~(pred condition#)))))
       `(get-entities ~entity-key))))

;; CAUTION!
;; データベース上の文字列の照合は、Javaにおける文字列の照合と同じであると仮定します。たとえば、大文字と小文字は異なる文字として扱われます。
;; LIKEでは「_」は使えません。「XXコードの3桁目が'A'の場合」などというクエリは、モデリング時に間違いがあった場合にのみ発生すると考えるためです。
;; BETWEENは使えません。「以上、かつ、以下」という仕様は、境界値が重複する恐れがあるので美しくないと考えるためです。
;; BOOLEANの場合でも、column = TRUE/FALSEと書いてください。パーサー作成（と実行）のコストを削減するためです。
;; 1対多の場合、多のエンティティのプロパティでの検索はできません。組織と社員の関係が1対多の場合、「給料が高い社員の組織」は検索できないことになります。「給料が高い社員の組織」は仕様として曖昧で危険なためです（給料が高い社員が1人でも所属している場合なのか、所属する全社員の給料が高い場合なのか、所属する社員の給料の平均が高い場合なのか分からない）。
;; 多対1の場合であっても、自己参照を2回以上繰り返している場合(= :boss.boss.name "X")などは、正しく動作しません。レア・ケースの割に、実装が大変なためです。そのような時は、SQLを指定するバージョンのget-entitiesを使用してください。

(defprotocol IMutableEntity
  (set-property!  [this property-key property-val])
  (delete-entity! [this]))

(defn property-informations
  [entity-key]
  (->> (base-entity-informations (get @entity-informations entity-key))
       (reverse)
       (map :property-informations)
       (map seq)
       (flatten)
       (apply array-map)))  ; I can't use (apply merge) for keeping order...

(defn add-to-cached-entities!
  [{entity-id :id entity-name :entity-name :as entity}]
  (let [entity-key      (keyword entity-name)
        root-entity-key (root-entity-key entity-key)]
    (letfn [(remove-has-many-property-cache []
              (doseq [has-many-property-cached? (->> (-> (property-informations entity-key)
                                                         (vals))
                                                     (filter #(= (:type %) :has-many))
                                                     (map #(keyword (str (name (:key %)) "-cached?"))))]
                (set-property! entity has-many-property-cached? false)))]
      (remove-has-many-property-cache)
      (swap! *cached-entities* assoc-in [root-entity-key entity-id] entity))))

(defn eval-option
  [entity-or-property-information key]
  (let [option-value (get-in entity-or-property-information [:options key])]
    (if (fn? option-value)
      (option-value)
      option-value)))

(defn create-entity!
  [entity-key fields]
  (let [entity-key            (or (keyword (:entity-name fields)) entity-key)
        property-informations (property-informations entity-key)
        fields                (atom (->> (cond-> fields
                                           (not (:id          fields)) (assoc :id          (:1 (first (jdbc/query *db* [(format "VALUES (NEXT VALUE FOR %s)" (as-quoted-sql-name :id-sequence))])))
                                                                              :inserted?   true)
                                           (not (:deleted?    fields)) (assoc :deleted?    false)
                                           (not (:entity-name fields)) (assoc :entity-name (name entity-key)))
                                         (mapcat (fn [[key value]]
                                                   [key (cond-> value
                                                          (= (class value) Boolean) (.booleanValue))]))  ; In Clojure, (Boolean. false) is true...
                                         (apply array-map)))
        database-fields       (->> (mapcat (fn [[property-key property-information]]
                                             (case (:type property-information)
                                               :belongs-to [(foreign-key-column-key property-key)]
                                               :has-many   nil
                                               :virtual    nil
                                               [property-key]))
                                           property-informations)
                                   (set))
        entity                (reify
                                ILookup
                                (valAt [this property-key]
                                  (let [property-information (get property-informations property-key)]
                                    (letfn [(get-belongs-to []
                                              (get-entity (:foreign-entity-key property-information) (get @fields (foreign-key-column-key (:key property-information))) :get-deleted? true))
                                            (get-has-many []
                                              (let [cached?                (keyword (str (name property-key) "-cached?"))
                                                    foreign-entity-key     (:foreign-entity-key property-information)
                                                    foreign-key-column-key (foreign-key-column-key (:foreign-property-key property-information))
                                                    foreign-key-column-val (:id @fields)]
                                                (if (get @fields cached?)
                                                  (->> (get-entities-from-cached-entities foreign-entity-key #(and (= (get % foreign-key-column-key) foreign-key-column-val) (not (:deleted? %))))
                                                       (sort-by :id))
                                                  (let [val (get-entities foreign-entity-key (= foreign-key-column-key foreign-key-column-val))]
                                                    (set-property! this cached? true)
                                                    val))))
                                            (get-virtual []
                                              ((:fn (:options property-information)) this))]
                                      (case (:type property-information)
                                        :belongs-to (get-belongs-to)
                                        :has-many   (get-has-many)
                                        :virtual    (get-virtual)
                                        (get @fields property-key)))))
                                (valAt [this property-key not-found]
                                  (let [val (.valAt this property-key)]
                                    (if (not (nil? val)) val not-found)))
                                IMutableEntity
                                (set-property! [this property-key property-val]
                                  (let [property-information (get property-informations property-key)]
                                    (letfn [(set-belongs-to! []
                                              (set-property! this (foreign-key-column-key property-key) (:id property-val)))
                                            (set-has-many! []
                                              (doseq [child-entity (get this property-key)]
                                                (set-property! child-entity (:foreign-property-key property-information) nil))
                                              (doseq [child-entity property-val]
                                                (set-property! child-entity (:foreign-property-key property-information) this)))]
                                      (case (:type property-information)
                                        :belongs-to (set-belongs-to!)
                                        :has-many   (set-has-many!)
                                        :virtual    (throw (IllegalStateException.))
                                        (do (swap! fields assoc property-key property-val)
                                            (when (contains? database-fields property-key)
                                              (swap! fields assoc :updated? true))))))
                                  this)
                                (delete-entity! [this]
                                  (let [deleted-entities (transient [this])]
                                    (doseq [composed-has-many-property (->> property-informations
                                                                            (vals)
                                                                            (filter #(and (= (:type %) :has-many)
                                                                                          (eval-option (get (hitokotonushi.entity/property-informations (:foreign-entity-key %)) (:foreign-property-key %)) :composition?))))]
                                      (doseq [composing-child-entity (get this (:key composed-has-many-property))]
                                        (doseq [deleted-entity (delete-entity! composing-child-entity)]
                                          (conj! deleted-entities deleted-entity))))
                                    (swap! fields assoc :deleted? true)
                                    (swap! fields assoc :updated? true)
                                    (persistent! deleted-entities))))]
    (letfn [(set-properties! []
              (doseq [[property-key property-val] (->> property-informations
                                                       (vals)
                                                       (mapcat #(let [property-type (:type %)
                                                                      property-key  (:key  %)]
                                                                  (if (or (= property-type :belongs-to)
                                                                          (= property-type :has-many))
                                                                    (if-let [property-val (get @fields property-key)]
                                                                      [[property-key property-val]])
                                                                    (if (nil? (get @fields property-key))
                                                                      (if-let [property-val (eval-option % :default)]
                                                                        [[property-key property-val]]))))))]
                (set-property! entity property-key property-val)))
            (add-to-cached-entities! []
              (hitokotonushi.entity/add-to-cached-entities! entity))]
      (set-properties!)
      (add-to-cached-entities!))
    entity))

(defn localized-label-string
  [property-information]
  (localized-string (eval-option property-information :label)))

(def ^:private valid-classes
  {:string    #{String}
   :text      #{String}
   :int       #{Long Integer}
   :decimal   #{BigDecimal Double Long Integer}
   :boolean   #{Boolean}
   :date      #{Date}
   :timestamp #{Date Timestamp}})
  
(defn validate-entity
  [entity]
  (letfn [(validate-property [{property-key :key field-type :field-type :as property-information}]
            (let [property-value   (get entity property-key)
                  string-property? (and field-type
                                        (or (= field-type :string)
                                            (= field-type :text)))]
              (letfn [(error-message [message & replacements]
                        (apply format (localized-string message) (localized-label-string property-information) replacements))
                      (validate-type []
                        (if (and field-type property-value)
                          (let [property-value-class (class property-value)]
                            (if-not (contains? (get valid-classes field-type) property-value-class)
                              [(error-message "item must be xxx type" (localized-string (str (name field-type) " type")))]))))
                      (validate-filled []
                        (if-not (eval-option property-information :optional?)
                          (if (if string-property?
                                (= (count property-value) 0)
                                (nil? property-value))
                            [(error-message "item is required")])))
                      (validate-size [compare-symbol message]
                        (let [compare-var (ns-resolve 'clojure.core compare-symbol)]
                          (if-let [compare-parameter (eval-option property-information (keyword (:name (meta compare-var))))]
                            (if-not (@compare-var (cond->> property-value
                                                    string-property? (count))
                                                  compare-parameter)
                              [(error-message (cond-> message
                                                string-property? (str " for string"))
                                              compare-parameter)]))))
                      (validate-less-than-or-equal-to []
                        (validate-size '<= "item must be less than or equal to"))
                      (validate-less-than []
                        (validate-size '<  "item must be less than"))
                      (validate-greater-than []
                        (validate-size '>  "item must be greater than"))
                      (validate-greater-than-or-equal-to []
                        (validate-size '>= "item must be greater than or equal to"))
                      (validate-re-matches []
                        (if-let [re (eval-option property-information :re-matches)]
                          (if-not (re-matches re property-value)
                            [(error-message "item is not correct")])))
                      (validate-by-validate-fn []
                        (if-let [validate-fn (get-in property-information [:options :validate])]
                          (if-let [message-and-replacements (validate-fn property-value entity)]
                            [(apply error-message message-and-replacements)])))]
                (if-let [error-messages (or (validate-type)
                                            (validate-filled)
                                            (not-empty (concat (validate-less-than-or-equal-to)
                                                               (validate-less-than)
                                                               (validate-greater-than-or-equal-to)
                                                               (validate-greater-than)
                                                               (validate-re-matches)
                                                               (validate-by-validate-fn))))]
                  [property-key error-messages]))))]
    (let [errors (apply array-map (mapcat validate-property
                                          (->> (property-informations (keyword (:entity-name entity)))
                                               (vals)
                                               (filter (fn [{property-type :type}]
                                                         (or (= property-type :belongs-to)
                                                             (= property-type :field)))))))]
      errors)))

(defn save!
  []
  (doseq [entities (vals @*cached-entities*)]
    (doseq [entity (vals entities)]
      (letfn [(column-and-values [entity-key root-entity-key]
                (let [sti? (not= entity-key root-entity-key)]
                  (cond-> (->> (property-informations entity-key)
                               (vals)
                               (mapcat #(let [property-type (:type %)]
                                          (cond
                                            (= property-type :belongs-to) (let [foreign-key-column-key (foreign-key-column-key (:key %))]
                                                                            [[foreign-key-column-key (get entity foreign-key-column-key)]])
                                            (= property-type :field)      (let [property-key (:key %)]
                                                                            [[property-key (get entity property-key)]]))))
                               (concat [[:deleted? (:deleted? entity)]]))
                    sti? (concat [[:entity-name (name entity-key)]]))))
              (optimistic-concurrency-control [execute-results]
                (when (not= execute-results [1])
                  (throw (SQLException. (localized-string "data that you tried to update/delete had been updated/deleted by another user")))))
              (update-cached-entity! [entity-key root-entity-key]
                (create-entity! entity-key (->> (jdbc/query *db*
                                                  [(format "SELECT * FROM %s WHERE (%s = ?)"
                                                           (as-quoted-sql-name root-entity-key)
                                                           (as-quoted-sql-name :id))
                                                   (:id entity)])
                                                (first))))
              (insert-entity! []
                (let [entity-key        (keyword (:entity-name entity))
                      root-entity-key   (root-entity-key entity-key)
                      column-and-values (column-and-values entity-key root-entity-key)]
                  (jdbc/execute! *db*
                    (cons (format "INSERT INTO %s (%s, %s, %s) VALUES (%s, ?, CURRENT_TIMESTAMP)"
                                  (as-quoted-sql-name root-entity-key)
                                  (->> column-and-values
                                       (map #(as-quoted-sql-name (first %)))
                                       (string/join ", "))
                                  (as-quoted-sql-name :id)
                                  (as-quoted-sql-name :modified-at)
                                  (->> (repeat (count column-and-values) "?")
                                       (string/join ", ")))
                          (concat (->> column-and-values
                                       (map second))
                                  [(:id entity)])))
                  (update-cached-entity! entity-key root-entity-key)))
              (update-entity! []
                (let [entity-key        (keyword (:entity-name entity))
                      root-entity-key   (root-entity-key entity-key)
                      column-and-values (column-and-values entity-key root-entity-key)]
                  (optimistic-concurrency-control (jdbc/execute! *db*
                                                    (cons (format "UPDATE %s SET %s, %s = CURRENT_TIMESTAMP WHERE (%s = ?) AND (%s = ?)"
                                                                  (as-quoted-sql-name root-entity-key)
                                                                  (->> column-and-values
                                                                       (map #(format "%s = ?" (as-quoted-sql-name (first %))))
                                                                       (string/join ", "))
                                                                  (as-quoted-sql-name :modified-at)
                                                                  (as-quoted-sql-name :id)
                                                                  (as-quoted-sql-name :modified-at))
                                                          (concat (->> column-and-values
                                                                       (map second))
                                                                  [(:id          entity)
                                                                   (:modified-at entity)]))))
                  (update-cached-entity! entity-key root-entity-key)))]
        (cond
          (:inserted? entity) (insert-entity!)
          (:updated?  entity) (update-entity!))))))
