(ns spongeblob.core
  (:require [spongeblob.protocols :as proto]
            [spongeblob.azure :refer [->AzureBlobStorage]]
            [spongeblob.s3 :refer [->S3BlobStorage]]))


(defn exists?
  "Check if a key exists in the storage."
  [store key]
  (proto/exists? store key))


(defn get-metadata
  "Fetches the object's metadata given a key"
  [store key]
  (proto/get-metadata store key))


(defn put-object
  "Store an object in the storage of choice under key."
  [store key content-type meta ^bytes bytes]
  (proto/put store key content-type meta bytes))


(defn get-object
  "Fetch an object from store given key.
   Returns a ByteArray."
  ^bytes [store key]
  (let [ret (proto/get store key)]
    (:data ret)))


(defn get-url
  "Get the URL for a given store and key."
  [store key]
  (proto/url store key))


(defn get-cdn-url
  "Get the CDN URL for a given store and key.
   If unavailable, return plain URL."
  [store key]
  (proto/cdn-url store key))


(defn azure-storage
  "Initialize Azure Blob Storage.

   Args:
   sas - Azure shared access signature.
   container - Name of container.
   base-uri - The base URL of the container.
   [OPTIONAL] cdn-uri - The base URL of the CDN."
  ([sas container base-uri]
   (azure-storage sas container base-uri nil))
  ([sas container base-uri cdn-uri]
   {:pre [(every? string? [sas container base-uri])
          (every? not-empty [sas container base-uri])
          (or (nil? cdn-uri)
              (and (string? cdn-uri)
                   (not-empty cdn-uri)))]}
   (->AzureBlobStorage sas container base-uri cdn-uri)))


(defn s3-storage
  "Initialize S3 Storage.

   Args:
   access-key - AWS access key id.
   secret-key - AWS secret access key id.
   bucket - Name of the bucket.
   [OPTIONAL] cloudfront-base-uri - The base URL of the CDN."
  ([access-key secret-key bucket]
   (s3-storage access-key secret-key bucket nil))
  ([access-key secret-key bucket cloudfront-base-uri]
   {:pre [(every? string? [access-key secret-key bucket])
          (every? not-empty [access-key secret-key bucket])
          (or (nil? cloudfront-base-uri)
              (and (string? cloudfront-base-uri)
                   (not-empty cloudfront-base-uri)))]}
   (->S3BlobStorage access-key secret-key bucket cloudfront-base-uri)))
