(ns spongeblob.core
  (:require [spongeblob.protocols :as proto]
            [spongeblob.util :refer [set-meta]]
            [spongeblob.azure :refer [->AzureBlobStorage]]
            [spongeblob.s3 :refer [->S3BlobStorage]]
            [spongeblob.fake-s3 :refer [->FakeS3BlobStorage]]
            [spongeblob.local-storage :refer [->LocalStorage]]))


(defn exists?
  "Check if a key exists in the storage."
  [store key]
  (proto/exists? store key))


(defn get-metadata
  "Fetches the object's metadata given a key"
  [store key]
  (proto/get-metadata store key))


(defn delete
  "Delete the object given the key"
  [store key]
  (proto/delete store key))


(defn put-object
  "Store an object in the storage of choice under key.
   Input:
   store - S3 store, Azure store etc.
   key - Key to access the object once uploaded
   content-type - Type of the content, ex. test/plain or application/json
                  default is application/octet-stream
   meta - Meta data for the object, any key value pair
   opts - :acl If acl given then set the acl for the object
               If acl not given then first put object then try to set :public-read acl
          :encryption If true then default encryption algorithm will be set which is \"AES256\"
                      If String then given encryption algorithm will be set
   bytes - Bytes array.
   Ex. (put-object s3store
                   \"test-data\"
                   \"text/plain\"
                   {\"one\" 1}
                   {:encryption true
                    :acl :private}
                   (.getBytes ^String \"Test-data\" \"UTF-8\") "
  ([store key meta ^bytes bytes]
   (put-object store key nil meta bytes))
  ([store key content-type meta ^bytes bytes]
   (proto/put store key content-type (set-meta meta {:encryption true}) bytes))
  ([store key content-type meta opts ^bytes bytes]
   (proto/put store
              key
              content-type
              (set-meta meta (merge {:encryption true} opts))
              bytes)))


(defn get-object
  "Fetch an object from store given key.
   Returns a ByteArray."
  ^bytes [store key]
  (let [ret (proto/get store key)]
    (:data ret)))


(defn get-object-stream
  "Fetch an object input stream from store given key.
   Returns a object input stream."
  [store key]
  (let [ret (proto/get-stream store key)]
    (:object-stream ret)))


(defn get-url
  "Get the URL for a given store and key."
  [store key]
  (proto/url store key))


(defn get-cdn-url
  "Get the CDN URL for a given store and key.
   If unavailable, return plain URL."
  [store key]
  (proto/cdn-url store key))


(defn azure-storage
  "Initialize Azure Blob Storage.

   Args:
   sas - Azure shared access signature.
   container - Name of container.
   base-uri - The base URL of the container.
   [OPTIONAL] cdn-uri - The base URL of the CDN."
  ([sas container base-uri]
   (azure-storage sas container base-uri nil))
  ([sas container base-uri cdn-uri]
   {:pre [(every? string? [sas container base-uri])
          (every? not-empty [sas container base-uri])
          (or (nil? cdn-uri)
              (and (string? cdn-uri)
                   (not-empty cdn-uri)))]}
   (->AzureBlobStorage sas container base-uri cdn-uri)))


(defn s3-storage
  "Initialize S3 Storage.

   Args:
   access-key - AWS access key id.
   secret-key - AWS secret access key id.
   bucket - Name of the bucket.
   [OPTIONAL] cloudfront-base-uri - The base URL of the CDN."
  ([access-key secret-key bucket]
   (s3-storage access-key secret-key bucket nil))
  ([access-key secret-key bucket cloudfront-base-uri]
   {:pre [(every? string? [access-key secret-key bucket])
          (every? not-empty [access-key secret-key bucket])
          (or (nil? cloudfront-base-uri)
              (and (string? cloudfront-base-uri)
                   (not-empty cloudfront-base-uri)))]}
   (->S3BlobStorage access-key secret-key bucket cloudfront-base-uri)))


(defn fake-s3-storage
  "Initialize fake S3 Storage.

   Args:
   endpoint - The fake s3 server endpoitn
   bucket - Name of the bucket"
  [endpoint bucket]
  (->FakeS3BlobStorage endpoint bucket))


(defn local-storage
  "Initialize a local storage.

   Blob store for local file storage. Useful in unit-tests.

   Args:
   directory           - Path of the local directory to store files.
   [OPTIONAL] base-uri - Endpoint for the independently running HTTP server. URL
                         otherwise will be the absolute path with `file://`
                         prefix."
  ([directory]
   (local-storage directory nil))
  ([directory base-uri]
   (->LocalStorage directory base-uri)))
