(ns spongeblob.s3
  (:require [clojure.java.io :as io]
            [spongeblob.util :refer [process-map acl]]
            [spongeblob.protocols :as proto :refer [BlobStore]]
            [clojure.string :refer [starts-with?]])
  (:import com.amazonaws.auth.BasicAWSCredentials
           com.amazonaws.services.s3.AmazonS3Client
           [com.amazonaws.services.s3.model
            AmazonS3Exception
            CannedAccessControlList
            ObjectMetadata
            PutObjectRequest
            DeleteObjectRequest
            S3Object
            S3ObjectInputStream]))

(declare s3-client make-put-request ->bytes)

(defrecord S3BlobStorage [access-key secret-key bucket cloudfront-uri])

(def canned-acl {:public-read CannedAccessControlList/PublicRead
                 :private CannedAccessControlList/Private})

(extend-type S3BlobStorage
  BlobStore
  (cdn-url [this key] (if-let [cf-uri (:cloudfront-uri this)]
                        (str cf-uri "/" key)
                        (proto/url this key)))

  (url [this key] (str "https://s3.amazonaws.com/" (:bucket this) "/" key))

  (get-metadata [this key]
    (try
      (let [^AmazonS3Client client (s3-client (:access-key this) (:secret-key this))
            ometa (.getObjectMetadata client (:bucket this) key)
            metadata (into {} (.getUserMetadata ometa))]
        (if-let [content-type (.getContentType ometa)]
          (assoc metadata "content-type" content-type)
          metadata))
      (catch AmazonS3Exception s3e
        (when-not (= (.getStatusCode s3e) 404)
          (throw s3e)))))

  (exists? [this key]
    (let [^AmazonS3Client client (s3-client (:access-key this) (:secret-key this))]
      (boolean
       (try (.getObjectMetadata client (:bucket this) key)
            (catch AmazonS3Exception s3e
              (when-not (= (.getStatusCode s3e) 404)
                (throw s3e)))))))

  (get-stream [this key]
    (try
      (let [^AmazonS3Client client (s3-client (:access-key this) (:secret-key this))
            ^S3Object res (.getObject client ^String (:bucket this) ^String key)]
        {:content-length (.getContentLength (.getObjectMetadata res))
         :content-type (.getContentType (.getObjectMetadata res))
         :object-stream (.getObjectContent res)})
      (catch AmazonS3Exception _
        (throw (ex-info (str "Key " key " doesn't exist!")
                        {:bucket (:bucket this)})))))

  (get [this key]
    (let [object-map (proto/get-stream this key)]
      (with-open [ois ^S3ObjectInputStream (:object-stream object-map)]
        (assoc (select-keys object-map [:content-type :content-length])
             :data (->bytes ois)))))

  (put [this key content-type meta bytes]
    (let [^AmazonS3Client client (s3-client (:access-key this) (:secret-key this))
          pr (make-put-request (:bucket this) key content-type meta bytes)]
      (.putObject client pr)
      (proto/url this key)))

  (delete [this key]
    (let [^AmazonS3Client client (s3-client (:access-key this) (:secret-key this))]
      (try (.deleteObject client (:bucket this) key)
           (catch AmazonS3Exception ex
             (if (starts-with? (.getMessage ex) "Access Denied")
               (throw (ex-info (str "You dont have enough permissions for deleting " key)
                               {:bucket (:bucket this)}))
               (throw (ex-info (.getMessage ex)
                               {:bucket (:bucket this)}))))))))


(defn ^:private s3-client*
  "Given access and secret keys, construct the client object."
  [access-key secret-key]
  (AmazonS3Client. (BasicAWSCredentials. access-key secret-key)))


(def ^{:doc "Create a AmazonS3Client obj given access & secret keys."} s3-client
  (memoize s3-client*))


(defn ^:private make-put-request
  "Construct a file upload request object."
  ^PutObjectRequest
  [bucket key content-type meta ^bytes bytes]
  (let [^java.util.Map metadata (-> meta
                                    (process-map :key-fn str :val-fn str))
        ^ObjectMetadata ometa
        (doto (ObjectMetadata.)
          (.setContentType (or content-type "application/octet-stream"))
          (.setCacheControl "max-age=3600, must-revalidate")
          (.setContentLength (count bytes))
          (.setUserMetadata metadata))
        acl-type (or (acl meta) :public-read)]
    (.withCannedAcl (PutObjectRequest. bucket key (io/input-stream bytes) ometa)
                    ^CannedAccessControlList (canned-acl acl-type))))


(defn ^:private ->bytes
  "Read an InputStream into a ByteArray."
  [is]
  (with-open [out (java.io.ByteArrayOutputStream.)]
    (clojure.java.io/copy (io/input-stream is) out)
    (.toByteArray out)))
