(ns spongeblob.local-storage
  "Blob store implementation for local files. Useful in unit-tests."
  {:author "Rohan <rohan@helpshift.com>"}
  (:require [clojure.edn :as ce]
            [clojure.java.io :as io]
            [clojure.string :as cs]
            [spongeblob.util :refer [process-map]]
            [spongeblob.protocols :as proto :refer [BlobStore]])
  (:import java.io.PushbackReader))

(declare mk-parent-dir get-url get-file-path get-meta-path)

(defrecord LocalStorage [directory base-uri])

(extend-type LocalStorage
  BlobStore
  (cdn-url [this key] (proto/url this key))

  (url [this key]
    (get-url this key))

  (get-metadata [this key]
    (when (proto/exists? this key)
      (->> (proto/url this key)
           get-meta-path
           io/reader
           PushbackReader.
           ce/read)))

  (exists? [this key]
    (.exists (io/file (get-file-path this key))))

  (get-stream [this key]
    (if (proto/exists? this key)
      (merge (proto/get-metadata this key)
             {:object-stream (io/reader (get-url this key))})
      (throw (ex-info (str "Key " key " doesn't exist!")
                      {:directory (:directory this)}))))

  (get [this key]
    (let [{:keys [object-stream] :as object-map} (proto/get-stream this key)]
      (merge (select-keys object-map [:content-type :content-length])
             {:data (slurp object-stream)})))

  (put [this key content-type meta ^bytes bytes]
    (let [meta (merge {}
                      meta
                      {"content-length" (count bytes)
                       "content-type" (or content-type
                                          "application/octet-stream")})
          file (get-file-path this key)
          meta-file (get-meta-path file)]
      (mk-parent-dir file)
      (with-open [w (io/output-stream file)]
        (.write w bytes))
      (with-open [w (io/writer meta-file)]
        (.write w (pr-str meta)))
      (proto/url this key))))


(defn- mk-parent-dir
  "Make parent directory for the given file path if non existent."
  [file-path]
  (let [dir (io/file (.getParent (io/file file-path)))]
    (or (.exists dir)
        (.mkdirs dir))))


(defn- get-url
  [store path]
  (let [base-uri (or (:base-uri store)
                     (str "file://" (:directory store)))]
    (str (cs/replace base-uri #"/$" "")
         "/"
         path)))


(defn- get-file-path
  "Get OS friendly file path"
  [store key]
  (str (io/file (:directory store)
                key)))


(defn- get-meta-path
  [file]
  (str file ".meta.edn"))


(defn delete-key
  "Delete a key from the store. Useful for doing a cleanup before tests."
  [store key]
  (let [file (get-file-path store key)
        meta (get-meta-path file)]
    (io/delete-file file)
    (io/delete-file meta)))
