(ns spongeblob.fake-s3
  (:require [clojure.java.io :as io]
            [clojure.string :as cs]
            [spongeblob.util :refer [process-map]]
            [spongeblob.protocols :as proto :refer [BlobStore]])
  (:import com.amazonaws.auth.BasicAWSCredentials
           com.amazonaws.services.s3.AmazonS3Client
           com.amazonaws.ClientConfiguration
           com.amazonaws.Protocol
           [com.amazonaws.services.s3.model AmazonS3Exception CannedAccessControlList ObjectMetadata PutObjectRequest S3Object DeleteObjectRequest]))

(declare fake-s3-client make-put-request ->bytes parse-endpoint)

(defrecord FakeS3BlobStorage [endpoint bucket])

(extend-type FakeS3BlobStorage
  BlobStore
  (cdn-url [this key] (proto/url this key))

  (url [this key]
    (let [{:keys [protocol host]} (parse-endpoint (:endpoint this))]
      (str protocol "://" (:bucket this) "." host "/" key)))

  (get-metadata [this key]
    (try
      (let [^AmazonS3Client client (fake-s3-client (:endpoint this))
            ometa (.getObjectMetadata client (:bucket this) key)
            metadata (into {} (.getUserMetadata ometa))]
        (if-let [content-type (.getContentType ometa)]
          (assoc metadata "content-type" content-type)
          metadata))
      (catch AmazonS3Exception s3e
        (when-not (= (.getStatusCode s3e) 404)
          (throw s3e)))))

  (exists? [this key]
    (let [^AmazonS3Client client (fake-s3-client (:endpoint this))]
      (boolean
       (try (.getObjectMetadata client (:bucket this) key)
            (catch AmazonS3Exception s3e
              (when-not (= (.getStatusCode s3e) 404)
                (throw s3e)))))))
  (get [this key]
    (let [^AmazonS3Client client (fake-s3-client (:endpoint this))]
      (try
        (with-open [^S3Object res (.getObject client ^String (:bucket this) ^String key)]
          {:content-length (.getContentLength (.getObjectMetadata res))
           :content-type (.getContentType (.getObjectMetadata res))
           :data (->bytes (.getObjectContent res))})
        (catch AmazonS3Exception _
          (throw (ex-info (str "Key " key " doesn't exist!")
                          {:bucket (:bucket this)}))))))
  (put [this key content-type meta bytes]
    (let [^AmazonS3Client client (fake-s3-client (:endpoint this))
          pr (make-put-request (:bucket this) key content-type meta bytes)]
      (.putObject client pr)
      (.getResourceUrl client (:bucket this) key))))


(defn ^:private fake-s3-client*
  "Given an endpoint for the fake s3 server construct the client object."
  [endpoint]
  (let [config (ClientConfiguration.)
        client (AmazonS3Client. (BasicAWSCredentials. "fake" "fake")
                                config)
        {:keys [protocol host]} (parse-endpoint endpoint)]
    (.setProtocol ^ClientConfiguration config
                  ({"http" Protocol/HTTP
                    "https" Protocol/HTTPS} protocol))
    (.setEndpoint ^AmazonS3Client client endpoint)
    client))


(def ^{:doc "Create a AmazonS3Client obj given access & secret keys."} fake-s3-client
  (memoize fake-s3-client*))


(defn ^:private make-put-request
  "Construct a file upload request object."
  ^PutObjectRequest
  [bucket key content-type meta ^bytes bytes]
  (let [^java.util.Map metadata (-> meta
                                    (process-map :key-fn str :val-fn str))
        ^ObjectMetadata ometa
        (doto (ObjectMetadata.)
          (.setContentType (or content-type "application/octet-stream"))
          (.setCacheControl "max-age=3600, must-revalidate")
          (.setContentLength (count bytes))
          (.setUserMetadata metadata))]
    (.withCannedAcl (PutObjectRequest. bucket key (io/input-stream bytes) ometa)
                    CannedAccessControlList/PublicRead)))


(defn ^:private ->bytes
  "Read an InputStream into a ByteArray."
  [is]
  (with-open [out (java.io.ByteArrayOutputStream.)]
    (clojure.java.io/copy (io/input-stream is) out)
    (.toByteArray out)))

(defn delete-key
  "Delete a key from the store. Useful for doing a cleanup before tests."
  [store key]
  (.deleteObject ^AmazonS3Client (fake-s3-client (:endpoint store))
                 (DeleteObjectRequest. (:bucket store) key)))

(defn parse-endpoint
  "Parse the protocol and host out of an endpoint URL."
  [endpoint]
  {:pre [endpoint]}
  (let [[protocol host] (cs/split endpoint (re-pattern "://"))]
    (if host
      {:protocol protocol
       :host host}
      {:protocol "http"
       :host protocol})))
