(ns daifu.core
  (:require [clojure.tools.cli :as cli]
            [clojure.tools.logging :as log]
            [daifu.diagnosis :as diagnosis]
            [daifu.core.middleware :as middleware]
            [daifu.core.visitation :as visitation]))

(defn conj-in
  "groups values together using the same key
   (conj-in {} :example :d1 #{})
   => {:example #{:d1}}
   (-> {}
       (conj-in :example :d1)
       (conj-in :example :d2))
   => {:example [:d1 :d2]}"
  {:added "0.2"}
  ([m k v]
   (conj-in m k v []))
  ([m k v initial]
   (update-in m [k] (fnil #(conj % v) initial))))

(def cli-options
  [["-c" "--checkup PATH" "Path to the checkups file that will be run"
    :default ".daifu"]
   ["-f" "--filter PATH"  "Path of files that filter"
    :assoc-fn #(conj-in %1 %2 %3 #{})]
   ["-i" "--indicator-path DIR" "Directory to load indicators"
    :assoc-fn conj-in]
   ["-t" "--target-path DIR" "Directory to load targets"
    :assoc-fn conj-in]
   ["-o" "--output FILE" "Path for output file"]
   [nil "--flatten" "Flattens the results into a vector"]
   [nil "--format FORMAT" "Format of output {edn|json}"
    :default :edn
    :parse-fn keyword
    :validate [#(#{:edn :json} %) "Must be a either edn or json"]]
   [nil "--no-defaults" "Do not load default indicators"]
   [nil "--pretty" "Pretty print the results"]
   [nil "--root PATH" "Path to the repository "
    :default (System/getProperty "user.dir")]
   [nil "--use-git" "Load files from git"]])

(defn process-single [visit [ik jk]]
  (let [jk   (or jk :default)
        indi (-> visit
                 :indicators
                 (get ik))
        target (-> visit
                 :targets
                 (get jk)
                 (merge (select-keys visit [:filter])))]
    (if (and indi target)
      (try
        ;;(println "started" (:id indi))
        (diagnosis/diagnose (:repository visit) indi target)
        (catch Throwable t
          (log/error t
                     (str "Failure for indicator" indi ", target" target)))))))

(defn process [{:keys [checkups] :as visit}]
  (->> checkups
       (keep (partial process-single visit))
       (zipmap (map first checkups))))

(defn -main [& args]
  (if-let [output ((->> cli/parse-opts
                        middleware/wrap-parse-errors
                        middleware/wrap-parse-help) args cli-options)]
    (-> (visitation/visitation (:options output))
        ((middleware/wrap-process-output process)))))

(comment
  (-main "--flatten" "--format" "json")
  (-main "--format" "json")
  )
