(ns daifu.diagnosis.summary
  (:require [daifu.diagnosis.result :as result]
            [daifu.diagnosis.summary.stat :as stat])
  (:import daifu.diagnosis.result.Result
           java.util.List
           clojure.lang.APersistentMap))

(defn group-distribute
  "distributes a flat list of results into a tree structure
   (group-distribute [{:ns 'hello :name 'a :value 1}
                      {:ns 'hello :name 'b :value 2}
                      {:ns 'world :name 'a :value 3}
                      {:ns 'world :name 'a :value 4}]
                     {:keys  [:ns :name]})
   => (contains-in
       {'hello {'a {:raw [{:value 1}]}
               'b {:raw [{:value 2}]}}
        'world {'a {:raw [{:value 3}
                          {:value 4}]}}})"
  {:added "0.2"}
  [results {:keys [keys] :as opts}]
  (reduce (fn [out result]
            (let [path (mapv #(get result %) keys)]
              (update-in out path (fnil (fn [res]
                                          (update-in res [:raw] conj (result/result result)))
                                        (result/result {:raw []})))))
          {}
          results))

(defn group-merge
  "calculates the stat for the group of results
   (group-merge {'hello {'a (result/result {:raw [(result/result {:value {}})]})
                         'b (result/result {:raw [(result/result {:value [{} {}]})]})}
                 'world {'a (result/result {:raw [(result/result {:value 3})
                                                  (result/result {:value 4})]})}}
                {:merge stat/average})
   => (contains-in
       {'hello {'a {:value 1}
               'b {:value 2}},
        'world {'a {:value 7/2}}})"
  {:added "0.2"}
  [groups {:keys [merge] :as opts}]
  (reduce-kv (fn [out k v]
               (if (result/result? v)
                 (->> (assoc v :value ((or merge stat/sum)
                                       (map stat/->stat (:raw v))))
                      (assoc out k))
                 (assoc out k (group-merge v opts))))
             {}
             groups))

(defn group
  "group results into a set of single statistics
   (group [{:ns 'hello :name 'a :value 1}
           {:ns 'hello :name 'b :value 2}
           {:ns 'world :name 'a :value 3}
           {:ns 'world :name 'a :value 4}]
          {:keys  [:ns :name]
           :merge stat/average})
   => (contains-in
      {'hello {'a {:value 1}
                'b {:value 2}}
        'world {'a {:value 7/2}}})"
  {:added "0.2"}
  [results opts]
  (-> results
      (group-distribute opts)
      (group-merge opts)))

(defn accumulate
  "accumulates results of the tree
   (-> (group [{:ns 'hello :name 'a :value 1}
               {:ns 'hello :name 'b :value 2}
               {:ns 'world :name 'a :value 3}
               {:ns 'world :name 'a :value 4}]
              {:keys  [:ns :name]
               :merge stat/sum})
       (accumulate [stat/average stat/average]))
  => (contains-in
       {:value 17/4
        :raw [{:value 3/2
               :raw [{:value 1}
                     {:value 2}]}
              {:value 7
               :raw [{:value 7}]}] })"
  {:added "0.2"}
  [groups [func & more]]
  (if (empty? more)
    (let [raw   (vec (vals groups))
          value (func (map stat/->stat raw))]
      (result/result {:value value :raw raw}))
    (accumulate (reduce-kv (fn [out k v]
                             (assoc out k (accumulate v more)))
                           {}
                           groups)
                [func])))
