(ns daifu.core.visitation
  (:require [daifu.diagnosis :as diagnosis]
            [daifu.diagnosis.indicator :as indicator]
            [daifu.diagnosis.target :as target]
            [gita.core :as git]
            [clojure.java.io :as io]))

(def +default-targets+
  {:default (target/target {:id :default
                            :type :project
                            ;:comparison true
                            ;:current {:commit "HEAD"}
                            ;:previous {:commit "HEAD^1"}
                            })})

(defrecord Visitation [])

(defmethod print-method Visitation
  [v w]
  (.write w (str "#visitation "
                 (into {} (-> v
                              (update-in [:indicators] (comp vec sort keys))
                              (update-in [:targets] (comp vec sort keys))
                              (dissoc :indicator-path :target-path))))))

(defn git-repo?
  "checks if repo is managed by git
   (git-repo? (io/file \".\"))
   => true"
  {:added "0.2"}
  [path]
  (.exists (io/file path ".git")))

(defn load-checkups
  "loads the checkups file (the default is '.daifu'), returns nil if not found.
   (load-checkups \".daifu\")
   => nil"
  {:added "0.2"}
  [path]
  (if (and path (.exists (io/file path)))
    (-> (slurp path) read-string)))

(defn visitation
  "constructs data regarding the daifu run
   (->> (visitation {:root \".\"})
        (into {}))
   => (contains {:root \".\",
                 :indicators map?,
                 :targets map?,
                 :checkups [[:arithmatic] [:collection] [:control]
                            [:equality] [:sequence] [:string]]
                :repository #(instance? java.io.File %)})"
  {:added "0.2"}
  [opts]
  (let [opts (if (:no-defaults opts)
               opts
               (update-in opts [:indicators] merge (indicator/load-defaults)))
        opts (update-in opts [:indicators]
                        merge (apply merge (map (fn [path]
                                                  (indicator/load-directory (str (:root opts) "/" path)))
                                                (:indicator-path opts))))        
        opts (update-in opts [:targets] merge +default-targets+)
        opts (update-in opts [:checkups]
                        #(->> (load-checkups (:checkup-path %))
                              vec))
        opts (if (empty? (:checkups opts))
               (assoc opts :checkups (vec (map vector (sort (keys (:indicators opts))))))
               opts)
        opts (assoc-in opts [:repository]
                       (if (and (git-repo? (:root opts)) (:use-git opts))
                         (git/repository (:root opts))
                         (io/file (:root opts))))]
    (map->Visitation opts)))
