(ns gram-api.hl
  (:require [gram-api.ll :as low-level]
            [clojure.tools.logging :as log]
            [clojure.data.json :as json]
            [clojure.string :as str]))

(defn try-read-
  "Tries to parse json document from string. If it fails it returns nil and logs error message "
  [json]
  (try (json/read-str json)
       (catch Exception e
         (do (log/warn "Failed to parse repsonse if sendMessage method")
             (log/debug e)
             nil))))


(defn next-update
  "Tries to get next message 10 times with delay 5000 ms. If it fails, it returns last error"
  [{:keys [api-key limit offset timeout] :or {limit 1 offset 1 timeout 10000} :as all}]
  (loop [attempt 10 last-error nil]
    (if (neg? attempt)
      (assoc all :error last-error)
      (let [response (low-level/get-updates api-key {:timeout timeout :limit limit :offset offset})
            body (:body response)
            json-body (try-read- body)
            last-offset (get-in json-body ["result" 0 "update_id"])]
        (if last-offset
          (if json-body
            (assoc all :offset (inc last-offset) :body json-body)
            (do (log/warn "next-update: Failed to parse result. Retrying in 5000 ms.")
                (Thread/sleep 5000)
                (recur (dec attempt) "next-update: Failed to parse result. See logs for more.")))
          (recur attempt last-error))))))

(defn send-message-cycled
  "Tries to send message to chat_id. If it fails 10 times, it'll return last error.
  Cannot detect 429 error. Use enqueue-message instead"
  [{:keys [api-key chat_id text parse_mode] :or {parse_mode "none"} :as all}]
  (loop [attempt 10 last-error nil]
    (if (neg? attempt)
      (assoc all :error last-error)
      (let [response (low-level/send-message api-key {:chat_id    chat_id :text text
                                                      :parse_mode parse_mode})
            body (:body response)
            json-body (try-read- body)
            error-code (get json-body "error_code")
            error-desc (get json-body "description")]
        (if (and json-body (not error-code))
          (assoc all :body json-body)
          (do (log/warnf "send-message-cycled: Error code: %s. %s" error-code error-desc)
              (Thread/sleep 5000)
              (recur (dec attempt) (format "send-message-cycled: Error code: %s. %s"
                                           error-code error-desc))))))))

(defn parse-int-
  "Extracts first number from given string. If it fails, return 0"
  [s]
  (Integer. ^String (or (re-find  #"\d+" (or s "0")) 0)))

(defn substr-vec-
  "Returns difference of substraction two vectors"
  [substrahent substractor]
  (vec (remove (into #{} substractor) substrahent)))

(defn queue-processor-
  "Returns queue processor that decides when to send messages from queue. May delay if
   it is not possible to send message right now"
  [_ watched _ {:keys [unsent next-retry api-key chat_id]}]
  (let [buffer-size (count unsent)
        we-take (max 5 (min 10 (int (Math/ceil (/ buffer-size 5)))))
        taken (take we-take unsent)
        now (System/currentTimeMillis)]
    (when (and (> now next-retry) (seq unsent))
      (let [combined-text (str/join "\n" taken)
            response (low-level/send-message api-key {:chat_id    chat_id :text combined-text
                                                      :parse_mode "Markdown"})
            body (:body response)
            json-body (try-read- body)
            error-code (get json-body "error_code")
            wait-on-429 (* (parse-int- (get json-body "description")) 1000)
            next-retry-on-429 (+ now wait-on-429)]
        (if (or (nil? json-body) (= error-code 429))
          (do (when (nil? json-body)
                (log/warn "queue-processor: Failed to parse result. Retrying in 5000 ms.")
                (swap! watched update-in [:next-retry] (fn [_ r] r) (+ now 5000))
                (future (Thread/sleep 5000) (swap! watched update-in [:next-retry] + 0)))
              (when (= error-code 429)
                (log/warnf "queue-processor: Got error 429. I'll retry in %s ms" wait-on-429)
                (swap! watched update-in [:next-retry] (fn [_ r] r) next-retry-on-429)
                (future (Thread/sleep wait-on-429) (swap! watched update-in [:next-retry] + 0))))
          (do (when (> buffer-size (count taken))
                (swap! watched update-in [:next-retry] (fn [_ r] r) (+ now 2000))
                (future (Thread/sleep 2000) (swap! watched update-in [:next-retry] + 0)))
              (swap! watched update-in [:unsent] substr-vec- taken)))))))

(defn prepare-queue-for-
  "Creates new message queue for given api-key and chat_id"
  [api-key chat_id]
  (let [result (atom {:unsent [] :next-retry 0 :api-key api-key :chat_id chat_id})]
    (add-watch result :queue-processor queue-processor-)
    result))

(defn enqueue-message
  "Puts message to queue. If queue is nil, it will be created for given api-key and chat_id.
  Returns input map with :queue replaced if it was required."
  [{:keys [api-key chat_id text queue] :as all}]
    (let [queue (or queue (prepare-queue-for- api-key chat_id))]
      (swap! queue update-in [:unsent] conj text)
      (assoc all :queue queue)))

(defn escape-markdown [text]
  (str/replace text #"[\\*\\_\[`]" #(str "\\" %)))
