(ns <<project-ns>>.product.product-events
  (:require
   ["antd" :as ant]
   ["uuid" :as uid]
   [clojure.string :as string]
   [kee-frame.core :as kf]
   [re-frame.core :as rf]
   [<<project-ns>>.url :refer [product-url]]
   [<<project-ns>>.product.events.custom-fabric-events :as fabric]
   [<<project-ns>>.product.events.custom-craftwork-events :as craft]
   [<<project-ns>>.product.events.custom-embroidery-events :as embroidery]
   [<<project-ns>>.product.product-init-db :refer [product-db]]
   [<<project-ns>>.product.product-data-convert :as convert]))

(kf/reg-controller
 :product/product-list-controller
 {:params (fn [route]
            (when (-> route :path-params :path (= "/product/list")) true))
  :start [:product/fetch-list nil]})

(kf/reg-controller
 :product/product-add-controller
 {:params (fn [route]
            (when (= "/product/info" (get-in route [:path-params :path] "")) true))
  :start [:request/get {:url (:categories product-url)
                        :callback-event :product/fetch-categories}]
  :stop [::reset]})

(kf/reg-controller
 :product/product-edit-controller
 {:params (fn [route]
            (let [path (get-in route [:path-params :path])
                  query (get route :query-string)]
              (when (and (= path "/product/info") (re-find #"id=\w+" (or query "")))
                (second (string/split query "=")))))
  :start (fn [_ id]
           [:product/fetch-detail id])})

;;商品列表
(kf/reg-chain
 :product/fetch-list
 (fn [{:keys [db]} [query]]
   (let [db-query (get-in db [:product :product-list :query] {})
         params (if query (merge db-query query) {:page 0 :size 10})]
     {:db (update-in db [:product :product-list]
                     #(assoc % :loading true :query params))
      :dispatch [:request/post {:url (:search product-url)
                                :params params
                                :callback-event :product/fetch-list-success}]})))

(kf/reg-event-fx
 :product/fetch-list-success
 (fn [{:keys [db]} [data query]]
   {:db (update-in db [:product :product-list]
                   #(merge % {:datas (:list data)
                              :loading false
                              :pagination {:total (:total data)
                                           :current (inc (:page query))
                                           :pageSize (:size query)}}))}))

;;商品详情
(defn recurse-category-tree [{:keys [sublist category_id category_name]}]
  (let [item {:value category_id
              :label category_name}]
    (if (and sublist (pos? (count sublist)))
      (assoc item :children (mapv recurse-category-tree sublist))
      (identity item))))

(kf/reg-event-fx
 :product/fetch-categories
 (fn [{:keys [db]} [categories]]
   {:db (assoc-in db [:product :categories] (mapv recurse-category-tree categories))}))

(kf/reg-event-fx
 ::reset
 (fn [{:keys [db]} [categories]]
   {:db (assoc-in db [:product :detail] (get-in product-db [:product :detail] {}))}))

(defn init-custom [{:keys [ids custom_craftworks]}]
  (let [category-id (second ids)
        style-ids (mapv :position_id (filter #(= "1" (:model_show %)) custom_craftworks))
        deep-ids (mapv :position_id (filter #(= "0" (:model_show %)) custom_craftworks))]
    (rf/dispatch [:product/fetch-fabrics {:category_id category-id}])
    (rf/dispatch [:product/fetch-positions {:category_id category-id}])
    (rf/dispatch [:product/fetch-embroideries {:category_id category-id}])
    (doseq [id style-ids]
      (rf/dispatch [:product/fetch-style-crafts {:category_id category-id :id id}]))
    (doseq [id deep-ids]
      (rf/dispatch [:product/fetch-deep-crafts {:category_id category-id :id id}]))))

(kf/reg-event-fx
 :product/fetch-detail
 (fn [{:keys [db]} [id]]
   {:db (assoc-in db [:product :detail :loading] true)
    :dispatch [:request/get {:url ((:detail product-url) id)
                             :callback-event :product/fetch-detail-success}]}))

(kf/reg-event-fx
 :product/fetch-detail-success
 (fn [{:keys [db]} [detail]]
   (init-custom detail)
   {:db (update-in db [:product :detail]
                   #(merge % (convert/product-dtail->db detail)))}))

(defn upload-imgs [imgs]
  (map
   (fn [img]
     (let [url (get-in img [:response :data :file_url])]
       (if url (assoc img :url url) img)))
   (filter #(not= (:status %) "removed") imgs)))

(defn update-images [image images]
  (map
   (fn [img]
     (if-let [url (get-in img [:response :data :file_url])]
       (assoc img :url url) img))
   (if-let [img (some #(= (:uid image) (:uid %)) images)]
     (if (= (:status image) "removed")
       (remove #(= (:uid %) (:uid image)) images)
       (map (fn [itm] (if (= (:uid itm) (:uid image)) image itm)) images))
     (into images [image]))))

(kf/reg-event-db
 :product/set-cover-imgs
 (fn [db [img]]
   (assoc-in db [:product :detail :cover-imgs]
             (if (not= (:status img) "removed")
               (if-let [url (get-in img [:response :data :file_url])]
                 [(assoc img :url url)]
                 [img])
               (identity [])))))

(kf/reg-event-db
 :product/set-banner-imgs
 (fn [db [img]]
   (update-in db [:product :detail :banner-imgs]
              #(update-images img (or % [])))))

(kf/reg-event-db
 :product/set-detail-imgs
 (fn [db [img]]
   (update-in db [:product :detail :detail-imgs]
              #(update-images img (or % [])))))

(kf/reg-event-db
 :product/set-current-step
 (fn [db [current-step]]
   (assoc-in db [:product :detail :current-step] current-step)))

(kf/reg-event-db
 :product/update-prodcut-detail
 (fn [db [prodcut-detail]]
   (assoc-in db [:product :detail]
             (merge (get-in db [:product :detail] {}) prodcut-detail))))

;;添加/更新商品
(kf/reg-chain
 :product/update
 (fn [{:keys [db]} [data]]
   {:db (update-in db [:product :detail] #(into {} [% data {:submit-loading false}]))})
 (fn [{:keys [db]}]
   (let [id (get-in db [:product :detail :product-id])
         url (if id ((:update product-url) id) (:add product-url))]
     {:dispatch [:request/post {:url url
                                :params (convert/product-db->request-params db)
                                :callback-event :product/update-success}]})))
(kf/reg-event-fx
 :product/update-success
 (fn [_ _]
   (.success ant/message "操作成功")
   {:dispatch [:core/nav :main {:path "/product/list"}]}))

;;商品删除
(kf/reg-event-fx
 :product/remove
 (fn [_ [id]]
   {:dispatch [:request/post {:url ((:remove product-url) id)
                              :callback-event :product/remove-success}]}))

(kf/reg-event-fx
 :product/remove-success
 (fn [{:keys [db]} _]
   (.success ant/message "删除成功")
   (let [query (get-in db [:product :product-list :query])]
     {:dispatch [:product/fetch-list query]})))

;;商品上下架
(kf/reg-chain
 :product/on-off
 (fn [_ [params]]
   {:dispatch [:request/post {:url (:on-off product-url)
                              :params params
                              :callback-event :product/do-success}]}))

;;商品推荐
(kf/reg-chain
 :product/recommend
 (fn [_ [params]]
   {:dispatch [:request/post {:url (:recommend product-url)
                              :params params
                              :callback-event :product/do-success}]}))

(kf/reg-event-fx
 :product/do-success
 (fn [{:keys [db]} _]
   (let [query (get-in db [:product :product-list :query])]
     {:dispatch [:product/fetch-list query]})))

(kf/reg-event-db
 :product/update-suit-products
 (fn [db [products]]
   (assoc-in db [:product :detail :suit-products] products)))

(kf/reg-event-db
 :product/remove-suite-product
 (fn [db [product]]
   (update-in db [:product :detail :suit-products]
              (fn [products]
                (remove #(= (:product_id product) (:product_id %)) products)))))

(kf/reg-event-db
 :product/show-suit-products
 (fn [db [show?]]
   (assoc-in db [:product :detail :suit-products-visible] show?)))

(kf/reg-event-db
 :product/set-product-custom?
 (fn [db [custom?]]
   (assoc-in db [:product :detail :product-custom] custom?)))
