(ns <<project-ns>>.components.search-product
  (:require
   ["antd" :as ant]
   [reagent.core :as r]
   [re-frame.core :as rf]
   [kee-frame.core :as kf]
   [<<project-ns>>.url :refer [product-url]]
   [<<project-ns>>.common.utils :as utils]))

(def FormItem (.-Item ant/Form))
(def SelectOption (.-Option ant/Select))

(def default-query
  {:page 0 :size 5})

(def columns [{:title "名称" :dataIndex "product_name"}
              {:title "编号" :dataIndex "product_no"}
              {:title "价格" :dataIndex "sales_price"}])

(def select-columns [{:title "名称"
                      :width 150
                      :dataIndex "product_name"}
                     {:title "价格" :dataIndex "sales_price"}
                     {:title "操作"
                      :render
                      (fn [val row]
                        (r/as-element
                         (let [values (js->clj row :keywordize-keys true)]
                           [:a
                            {:on-click
                             (fn []
                               (rf/dispatch [::update-select-data values false]))} "移除"])))}])

(defn table-search []
  (utils/create-form
   (fn [props]
     (let [this (utils/get-form)]
       [:> ant/Form {:layout "inline"
                     :onSubmit (fn [e]
                                 (.preventDefault e)
                                 ((:validateFields this)
                                  (fn [err values]
                                    (rf/dispatch [::fetch-data
                                                  (merge default-query
                                                         (into {} (remove (fn [[k v]] (nil? v)) (js->clj values :keywordize-keys true))))]))))}
        [:> ant/Row {:style {:margin-bottom 10}}
         [:> ant/Col {:md 8}
          [:> FormItem
           (utils/decorate-field
            this "product_name"
            [:> ant/Input {:allowClear true
                           :placeholder "商品名称"}])]]
         [:> ant/Col {:md 8}
          [:> FormItem
           (utils/decorate-field
            this "product_no"
            [:> ant/Input {:allowClear true
                           :placeholder "商品编号"}])]]
         [:> ant/Col {:md 5}
          [:> FormItem
           [:> ant/Button {:type "primary" :htmlType "submit"} "查询"]]]]]))))

(defn search-product-view [{:keys [on-ok]}]
  (def search-product (rf/subscribe [::search-product]))
  (def select-data-ids (rf/subscribe [::select-data-ids]))
  (fn []
    [:> ant/Modal {:title "选择商品"
                   :width 900
                   :style {:top 20}
                   :closable false
                   :visible (:visible @search-product)
                   :onCancel #(rf/dispatch [::close])
                   :onOk (fn []
                           (on-ok (:select-data @search-product))
                           (rf/dispatch [::close]))}
     [:> ant/Row
      [:> ant/Col {:span 13}
       [table-search]]]
     [:> ant/Row
      [:> ant/Col {:span 13}
       [:> ant/Table
        {:rowKey "product_id"
         :columns columns
         :size "small"
         :loading (or (:loading @search-product) false)
         :dataSource (or (:data @search-product) [])
         :pagination (or (:pagination @search-product) {})
         :onChange (fn [pagination]
                     (let [{:keys [pageSize current]} (js->clj pagination :keywordize-keys true)]
                       (rf/dispatch [::fetch-data {:page (dec current) :size pageSize}])))
         :rowSelection {:selectedRowKeys @select-data-ids
                        :onSelect (fn [record selected]
                                    (rf/dispatch [::update-select-data (js->clj record :keywordize-keys true) selected]))}}]]
      [:> ant/Col {:span 10 :offset 1}
       [:> ant/Table
        {:columns select-columns
         :title (fn [] "选中的商品")
         :showHeader false
         :size "small"
         :pagination false
         :dataSource (:select-data @search-product)}]]]]))

(kf/reg-chain
 :search-product/open
 (fn [{:keys [db]} [select-data query]]
   {:db (update-in db [:common :search-product] #(merge % {:visible true :select-data (or select-data [])}))
    :dispatch [::fetch-data (merge default-query (or query {}))]}))

(kf/reg-chain
 ::close
 (fn [{:keys [db]}]
   {:db (assoc-in db [:common :search-product :visible] false)}))

(kf/reg-chain
 ::fetch-data
 (fn [{:keys [db]} [query]]
   (let [new-query (merge (get-in db [:common :search-product :query]) query)]
     {:db (assoc-in db [:common :search-product :loading] true)
      :dispatch [:request/post {:url (:search product-url)
                                :params new-query
                                :callback-event :search-product/update-data}]})))

(kf/reg-event-db
 :search-product/update-data
 (fn [db [{:keys [total list]} {:keys [page size] :as query}]]
   (update-in db [:common :search-product]
              #(merge % {:data list
                         :query query
                         :loading false
                         :pagination {:current (inc page)
                                      :total total
                                      :pageSize size}}))))

(kf/reg-event-db
 ::update-select-data
 (fn [db [record selected?]]
   (update-in
    db
    [:common :search-product :select-data]
    (fn [select-data]
      (if selected?
        (if (some #(= (:product_id %) (:product_id record)) select-data)
          (identity select-data)
          (concat select-data [record]))
        (remove #(= (:product_id %) (:product_id record)) select-data))))))

;;查询商品所有数据
(rf/reg-sub
 ::search-product
 (fn [db]
   (get-in db [:common :search-product] {})))

;;选择商品数据
(rf/reg-sub
 ::select-data
 :<- [::search-product]
 (fn [search-product]
   (get search-product :select-data [])))

;;选择商品id列表
(rf/reg-sub
 ::select-data-ids
 :<- [::select-data]
 (fn [select-data]
   (map #(:product_id %) select-data)))
