(ns {{project-ns}}.product.product-info-views
  (:require
   ["antd" :as ant]
   [reagent.core :as r]
   [re-frame.core :as rf]
   [{{project-ns}}.common.utils :as utils]
   [{{project-ns}}.product.product-events]
   [{{project-ns}}.components.upload :refer [file-upload]]
   [{{project-ns}}.components.common-page :refer [main-page]]))

(def FormItem (.-Item ant/Form))
(def TextArea (.-TextArea ant/Input))
(def RadioGroup (.-Group ant/Radio))
(def TreeNode (.-TreeNode ant/TreeSelect))
(def Step (.-Step ant/Steps))

(declare product-info-form)
(declare product-status-form)

(def step-froms [#'product-info-form #'product-status-form])

(def product-detail (rf/subscribe [:product/detail]))

(defn product-info-page []
  (fn []
    [main-page {:title
                (if (:product-id @product-detail) "更新商品信息" "添加商品信息")
                :on-back #(rf/dispatch [:core/nav :main {:path "/product/list"}])}
     [:div
      [:> ant/Row
       [:> ant/Col {:span 20 :offset 3}
        [:> ant/Spin {:spinning (:loading @product-detail)}
         [:> ant/Row {:style {:margin-bottom 30}}
          [:> ant/Col {:span 12 :offset 3}
           [:> ant/Steps {:current (:current-step @product-detail)
                          :size "small"}
            [:> Step {:title "商品信息"}]
            [:> Step {:title "商品状态"}]]
           ]]
         [(step-froms (:current-step @product-detail))]]]]]]))

(declare one-col-item)
(declare two-col-item)
(declare three-col-item)
(declare parameter-input)
(declare step-btn)

(defn product-info-form []
  (utils/create-form
   (fn [props]
     (let [form (utils/get-form)]
       [:span
        (utils/decorate-field
         form "product-id"
         {:initialValue (:product-id @product-detail)}
         [:span])]
       [:> ant/Form
        [two-col-item
         "商品名称"
         (utils/decorate-field
          form "product-name"
          {:rules [{:required true
                    :message "请输入商品名称"}]
           :initialValue (:product-name @product-detail)}
          [:> ant/Input
           {:style {:width 200}
            :placeholder "请选输入商品名称"}])
         "商品编码"
         (utils/decorate-field
          form "product-no"
          {:rules [{:required true
                    :message "请输入商品编码"}]
           :initialValue (:product-no @product-detail)}
          [:> ant/Input
           {:style {:width 200}
            :placeholder "请选输入商品编码"}])]

        [two-col-item
         "商品品类:"
         (utils/decorate-field
          form "category-ids"
          {:rules [{:required true
                    :message "请选择商品品类"}]
           :initialValue (:category-ids @product-detail)}
          [:> ant/Cascader
           {:style {:width 200}
            :placeholder "请选择品类"
            :options @(rf/subscribe [:product/categories])}])
         "商品套装:"
         (utils/decorate-field
          form  "suit-type"
          {:rules [{:required true
                    :message "请输入商品编码"}]
           :initialValue (:suit-type @product-detail)}
          [:> RadioGroup
           [:> ant/Radio {:value "0"} "单品"]
           [:> ant/Radio {:value "1"} "套装"]])]

        [two-col-item
         "商品价格:"
         (utils/decorate-field
          form "sale-price"
          {:rules [{:required true
                    :message "请输入商品编码"}]
           :initialValue (:sale-price @product-detail)}
          [:> ant/InputNumber {:placeholder "请输入商品价格" :min 0 :style {:width 200}}])
         "会员价格:"
         (utils/decorate-field
          form "vip-price"
          {:rules [{:required true
                    :message "请输入商品编码"}]
           :initialValue (:vip-price @product-detail)}
          [:> ant/InputNumber {:placeholder "请输入会员价格" :min 0 :style {:width 200}}])]

        [one-col-item
         "规格参数:"
         [parameter-input
          {:data @(rf/subscribe [:product/detail-params])
           :on-plus #(rf/dispatch [:product/add-detail-param {:key "" :value ""}])
           :on-minus #((rf/dispatch [:product/remove-detail-params %]))
           :on-key-change #(rf/dispatch [:product/update-detail-params %])
           :on-value-change #(rf/dispatch [:product/update-detail-params %])}]]

        [one-col-item
         "服务参数:"
         [parameter-input
          {:data @(rf/subscribe [:product/detail-services])
           :on-plus #(rf/dispatch [:product/add-detail-service {:key "" :value ""}])
           :on-minus #((rf/dispatch [:product/remove-detail-service %]))
           :on-key-change #(rf/dispatch [:product/update-detail-service %])
           :on-value-change #(rf/dispatch [:product/update-detail-service %])}]]

        [one-col-item
         "商品简介"
         (utils/decorate-field
          form "product-intro"
          {:initialValue (:product-intro @product-detail)}
          [:> TextArea {:placeholder "请输入商品简介"
                        :style {:width 420
                                :height 100}}])]

        [one-col-item
         "封面图片:"
         [file-upload
          {:files @(rf/subscribe [:product/detail-cover-imgs])
           :data {:file_prefix "product/cover"}
           :on-change (fn [{:keys [file]}]
                        (rf/dispatch [:product/set-cover-imgs [file]]))}]]

        [one-col-item
         "轮播图片:"
         [file-upload
          {:multi? true
           :files @(rf/subscribe [:product/detail-banner-imgs])
           :data {:file_prefix "product/banner"}
           :on-change (fn [{:keys [fileList]}]
                        (rf/dispatch [:product/set-banner-imgs fileList]))}]]

        [one-col-item
         "详细图片:"
         [file-upload
          {:multi? true
           :files @(rf/subscribe [:product/detail-detail-imgs])
           :data {:file_prefix "product/detail"}
           :on-change (fn [{:keys [fileList]}]
                        (rf/dispatch [:product/set-detail-imgs fileList]))}]]

        [step-btn form]]))))

(defn product-status-form []
  (fn []
    (utils/create-form
     (fn [props]
       (let [this (utils/get-form)]
         [:> ant/Form
          [three-col-item
           "上下架"
           (utils/decorate-field
            this "online?"
            {:valuePropName "checked"
             :initialValue (:online? @product-detail)}
            [:> ant/Switch
             {:checkedChildren "上架"
              :unCheckedChildren "下架"}])

           "是否新品"
           (utils/decorate-field
            this "new?"
            {:valuePropName "checked"
             :initialValue (:new? @product-detail)}
            [:> ant/Switch
             {:checkedChildren "是"
              :unCheckedChildren "否"}])

           "是否推荐"
           (utils/decorate-field
            this "recommend?"
            {:valuePropName "checked"
             :initialValue (:recommend? @product-detail)}
            [:> ant/Switch
             {:checkedChildren "推荐"
              :unCheckedChildren "不推荐"}])]

          [step-btn this]])))))

(defn step-btn [form]
  (let [current-step (:current-step @product-detail)
        submit-loading (:submit-loading @product-detail)]
    [:> ant/Row
     [:> ant/Col {:span 20 :offset 3}
      (when (pos? current-step)
        [:> ant/Button {:style
                        {:margin-right 15}
                        :on-click
                        (fn []
                          (rf/dispatch [:product/set-current-step (dec current-step)]))} "上一步"])

      (when (< current-step (dec (count step-froms)))
        [:> ant/Button {:style {:margin-right 15}
                        :type "primary"
                        :on-click
                        (fn []
                          ((:validateFieldsAndScroll form)
                           (fn [err values]
                             (when (not err)
                               (rf/dispatch [:product/update-prodcut-detail
                                             (js->clj values :keywordize-keys true)])
                               (rf/dispatch [:product/set-current-step (inc current-step)])))))} "下一步"])

      (when (= current-step (dec (count step-froms)))
        [:> ant/Button {:style {:margin-right 15}
                        :type "primary"
                        :loading submit-loading
                        :on-click
                        (fn []
                          ((:validateFieldsAndScroll form)
                           (fn [err values]
                             (when (not err)
                               (rf/dispatch [:product/update-product
                                             (js->clj values :keywordize-keys true)])))))} "保存 "])]]))

(defn one-col-item [title content]
  (let [props {:label title
               :style {:margin-left 10}
               :labelCol {:span 3}
               :wrapperCol {:span 18}}]
    [:> ant/Row
     [:> ant/Col {:span 18}
      [:> FormItem props
       content]]]))

(defn two-col-item []
  (defn two-item-props [title]
    {:label title :labelCol {:span 6} :wrapperCol {:span 18}})
  (fn []
    (let [this (r/current-component)
          children (r/children this)]
      [:> ant/Row
       [:> ant/Col {:span 10}
        [:> FormItem (two-item-props (get children 0))
         (get children 1)]]
       [:> ant/Col {:span 10}
        [:> FormItem (two-item-props (get children 2))
         (get children 3)]]])))

(defn three-col-item []
  (defn three-item-props [title]
    {:style {:margin-left 10}
     :label title
     :labelCol {:span 7}
     :wrapperCol {:span 15}})
  (fn []
    (let [this (r/current-component)
          children (r/children this)]
      [:> ant/Row
       [:> ant/Col {:span 8}
        [:> FormItem (three-item-props (get children 0))
         (get children 1)]]
       [:> ant/Col {:span 8}
        [:> FormItem (three-item-props (get children 2))
         (get children 3)]]
       [:> ant/Col {:span 8}
        [:> FormItem (three-item-props (get children 4))
         (get children 5)]]])))

(defn parameter-input []
  (let [{:keys [data on-plus on-minus on-key-change on-value-change]} (r/props (r/current-component))]
    [:div
     (when (or (nil? data) (zero? (count data)))
       [:> ant/Button {:ghost true
                       :type "primary"
                       :shape "circle"
                       :icon "plus"
                       :size "small"
                       :on-click #(on-plus)}])
     (map-indexed
      (fn [idx itm]
        [:div {:key idx}
         [:> ant/Input {:value (get itm :key "")
                        :placeholder "名称"
                        :allowClear true
                        :style {:width 180}
                        :onChange #(on-key-change (assoc itm :key (-> % .-target .-value)))}]
         [:> ant/Input {:value (get itm :value "")
                        :placeholder "值"
                        :allowClear true
                        :style {:width 180 :margin "0 15px"}
                        :onChange #(on-value-change (assoc itm :value (-> % .-target .-value)))}]
         (when (and data (> (count data) 1))
           [:> ant/Button {:style {:margin-right 10}
                           :ghost true
                           :type "primary"
                           :shape "circle"
                           :icon "minus"
                           :size "small"
                           :on-click #(on-minus itm)}])
         (when (or (nil? itm) (= (inc idx) (count data)))
           [:> ant/Button {:ghost true
                           :type "primary"
                           :shape "circle"
                           :icon "plus"
                           :size "small"
                           :on-click #(on-plus)}])])
      data)]))
