(ns hara.platform.mail.remote
  (:require [hara.protocol.mail :as protocol.mail]
            [hara.protocol.component :as protocol.component]
            [hara.platform.mail.interop.message :as message]
            [hara.core.component :as component])
  (:import java.util.Properties
           javax.mail.Session
           javax.mail.Folder)
  (:refer-clojure :exclude [send]))

(def ^:dynamic *default-ports*
  {:smtp {:none 25
          :ssl 465}
   :pop3 {:none 110
          :ssl 995}
   :imap {:none 143
          :ssl 993}})

(def access-map {:read-only  Folder/READ_ONLY
                 :read-write Folder/READ_WRITE})

(def ^:dynamic *default-folder* "INBOX")

(def ^:dynamic *default-access* :read-write)

(defmethod print-method javax.mail.Service
  [v w]
  (.write w (str "#service " "\"" (str (.getURLName v)) "\"")))

(defmethod print-method javax.mail.Folder
  [v w]
  (.write w (str "#folder " {:count (.getMessageCount v)
                             :name (.getFullName v)})))

(defn get-transport
  "gets the transport for a particular session
 
   (get-transport (Session/getInstance (Properties.))
                  \"smtp\")
   ;; #service \"smtps://\"
   => com.sun.mail.smtp.SMTPTransport"
  {:added "3.0"}
  [session protocol]
  (.getTransport session protocol))

(defn get-store
  "gets the mailbox store for a particular session
 
   (get-store (Session/getInstance (Properties.))
              \"imap\")
   ;;=> #service \"imap://...\"
   => com.sun.mail.imap.IMAPStore"
  {:added "3.0"}
  [session protocol]
  (.getStore session protocol))

(defn connect
  "creates a service given method and settings
   
   (connect -test-smtp- get-transport)
   ;;=> #service \"smtps://...\"
   => com.sun.mail.smtp.SMTPSSLTransport"
  {:added "3.0"}
  [{:keys [type host port security auth params]} get-service-fn]
  (let [security (or security :none)
        session  (Session/getDefaultInstance (or params
                                                 (Properties.)))
        protocol (str (name type)
                      (if (= :none security) "" "s"))
        service (get-service-fn session protocol)
        port    (or port (-> *default-ports* type security))]
    (if auth
      (.connect service host port (:username auth) (:password auth))
      (.connect service host port))
    service))

(defn send
  "creates a service given method and settings
   
   (send -test-smtp- (config/resolve
                      {:from [:env \"TEST_HARA_MAILBOX_ID\"]
                       :to   [[:env \"TEST_HARA_MAILBOX_ID\"]]
                       :subject \"Test\"
                      :body \"Hello There\"}))"
  {:added "3.0"}
  [mailer message]
  (let [conn (connect mailer get-transport)
        m (message/message message)]
    (.sendMessage conn m (.getAllRecipients m))))

(defn send-bulk
  "creates a service given method and settings
   
   (send-bulk -test-smtp-
              (config/resolve
               [{:to   [[:env \"TEST_HARA_MAILBOX_ID\"]]
                 :subject \"Test A\"
                :body \"Hello There A\"}
                {:to   [[:env \"TEST_HARA_MAILBOX_ID\"]]
                 :subject \"Test B\"
                 :body \"Hello There B\"}]))"
  {:added "3.0"}
  [mailer messages]
  (let [conn (connect mailer get-transport)]
    (doseq [message messages]
      (let [m (message/message message)]
        (.sendMessage conn m (.getAllRecipients m))))))

(defn remote-folder
  "returns the folder for message retrieval
   
   (type (remote-folder (connect -test-imap-
                                 get-store)
                        \"INBOX\"
                        :read-write))
   ;; #folder {:count <>, :name \"INBOX\"}
   => com.sun.mail.imap.IMAPFolder"
  {:added "3.0"}
  ([connection]
   (remote-folder connection nil nil))
  ([connection name access]
   (let [folder (.getFolder connection (or name
                                           *default-folder*))
         _ (.open folder (access-map  (or access
                                          *default-access*)))]
     folder)))

(defn remote-state
  "returns both the storage and folder connection
 
   (remote-state (config/resolve
                  {:type :pop3
                   :host [:env \"TEST_HARA_MAILBOX_MAIL\"]
                   :folder   \"INBOX\"    ;; default
                   :access   :read-only ;; default :read-write
                   :security :none      ;; default
                   :auth -test-auth-}))
   ;; {:connection #service \"pop3://...\"
   ;;  :folder #folder {:count <>, :name \"INBOX\"}}
   => (contains {:connection com.sun.mail.pop3.POP3Store
                 :folder com.sun.mail.pop3.POP3Folder})"
  {:added "3.0"}
  [mailbox]
  (let [connection (connect mailbox get-store)
        folder (remote-folder connection
                              (:folder mailbox)
                              (:access mailbox))]
    {:connection connection :folder folder}))

(defn mailbox-folder
  "returns the folder for message retrieval, contains connection state
 
   (def -box- (-> (mail/create-mailbox
                   (config/resolve {:type :pop3
                                    :host [:env \"TEST_HARA_MAILBOX_MAIL\"]
                                    :auth -test-auth-}))
                  (component/start)))
 
   (time (mailbox-folder -box-)) ;; initial
   
   (time (mailbox-folder -box-)) ;; faster than before
   => com.sun.mail.pop3.POP3Folder"
  {:added "3.0"}
  [{:keys [state] :as mailbox}]
  (let [{:keys [folder connection]} @state]
    (cond (or (nil? connection)
              (not (.isConnected connection)))
          (let [rs (remote-state mailbox)]
            (-> (reset! state rs)
                :folder))

          (and folder (.isOpen folder))
          folder

          (.isConnected connection)
          (let [folder (remote-folder connection
                                      (:folder mailbox)
                                      (:access mailbox))]
            (-> (swap! state assoc :folder folder)
                :folder)))))

(defrecord RemoteMailer []
  Object
  (toString [v]
    (str "#mailer" (into {} (dissoc v :state))))

  protocol.mail/IMailer
  (-send-mail [mailer message]
    (send mailer message))

  (-send-bulk [mailer messages]
    (send-bulk mailer messages)))

(defmethod print-method RemoteMailer
  [v w]
  (.write w (str v)))

(defmethod protocol.mail/-create-mailer :smtp
  [m]
  (map->RemoteMailer m))

(defrecord RemoteMailbox []
  Object
  (toString [v]
    (str "#mailbox" (into {} (dissoc v :state))))

  protocol.mail/IMailbox
  (-count-mail  [mailbox]
    (.getMessageCount (mailbox-folder mailbox)))
  (-get-mail    [mailbox i]
    (.getMessage (mailbox-folder mailbox)))
  (-list-mail   [mailbox]
    (.getMessages (mailbox-folder mailbox)))
  (-clear-mail  [mailbox])

  protocol.component/IComponent
  (-start [mailbox]
    (assoc mailbox :state (atom {})))

  (-stop [mailbox]
    (dissoc mailbox :state)))

(defmethod print-method RemoteMailbox
  [v w]
  (.write w (str v)))

(defmethod protocol.mail/-create-mailbox :imap
  [m]
  (map->RemoteMailbox m))

(defmethod protocol.mail/-create-mailbox :pop3
  [m]
  (map->RemoteMailbox m))
