(ns hara.platform.table.sql
  (:require [hara.lib.jackson :as json]
            [hara.string :as string]
            [hara.core.base.util :as util]))

(def ^:dynamic *raw* nil)

(def ^:dynamic *enum->str* nil)

(defn format-input
  "formats input given options
 
   (format-input {} :meat {:type :beef :amount 1 :grade \"AA\"})
   => {:type :beef, :amount 1, :grade \"AA\"}
 
   (format-input {:format {:meat {:type :edn}}} :meat {:type :beef :amount 1 :grade \"AA\"})
   => {:type \":beef\", :amount 1, :grade \"AA\"}"
  {:added "3.0"}
  ([{:keys [format] :as store} table data]
   (reduce-kv (fn [out k type]
                (if-let [v  (get data k)]
                  (assoc out k (case type
                                 :edn   (pr-str v)
                                 :json  (json/write v)))
                  out))
              data
              (get format table))))

(defn format-output
  "formats output given options
   
   (format-output {:format {:meat {:type :edn}}}
                  :meat
                  {:type \":beef\", :amount 1, :grade \"AA\"})
   => {:type :beef :amount 1 :grade \"AA\"}"
  {:added "3.0"}
  ([{:keys [format] :as store} table data]
   (reduce-kv (fn [out k type]
                (if-let [v  (get data k)]
                  (assoc out k (case type
                                 :edn   (read-string v)
                                 :json  (json/read v json/+keyword-mapper+)))
                  (dissoc out k)))
              data
              (get format table))))

(defn ref-input
  "converts a ref keyword into sql column
 
   (ref-input {:schema -schema-}
              :account
              {:id \"account-0\" :wallet :wallet.id/w0})
   => {:id \"account-0\", :wallet-id \"w0\"}"
  {:added "3.0"}
  ([{:keys [schema] :as store} table data]
   (let [columns (get (:tree schema) table)]
     (reduce-kv (fn [out k v]
                  (if-not (nil? v)
                    (let [[{:keys [type] :as attr}] (get columns k)
                          [k v] (cond (= type :ref)
                                      (if (keyword? v)
                                        [(keyword (str (name k) "-id"))
                                         (name v)]
                                        [k v])
                                      
                                      (= type :alias) nil

                                      (= type :enum) [k ((or *enum->str* name) v)]
                                      
                                      :else [k v])]
                      (cond-> out k (assoc k v)))
                    out))
                {}
                data))))

(defn ref-output
  "converts sql columns into a ref keyword
 
   (ref-output {:schema -schema-}
               :account
               {:id \"account-0\" :wallet-id \"w0\"})
   => {:id \"account-0\", :wallet :wallet.id/w0}"
  {:added "3.0"}
  ([{:keys [schema] :as store} table data]
   (let [columns (get (:tree schema) table)]
     (reduce-kv (fn [out k v]
                  (if-not (nil? v)
                    (let [sk (name k)
                          [nk nv] (cond (.endsWith sk "-id")
                                        (let [nk (keyword (subs sk 0 (- (count sk) 3)))
                                              [{:keys [type] :as attr}] (get columns nk)]
                                          (if (= type :ref)
                                            [nk (keyword (str (-> attr :ref :ns name) ".id") v)]
                                            [k v]))
                                        
                                        :else
                                        (let [[{:keys [type] :as attr}] (get columns k)]
                                          (cond (= type :enum)
                                                [k (keyword v)]
                                                
                                                :else
                                                [k v])))]
                      (assoc out nk nv))
                    out))
                {}
                data))))

(defn alias-to-string
  "converts a value to string
 
   (alias-to-string :ref :wallet.id/w0)
   => \"w0\"
 
   (alias-to-string :enum :trade)
   => \"trade\""
  {:added "3.0"}
  ([type value]
   (case type
     :ref  (name value)
     :enum ((or *enum->str* name) value)
     (str value))))

(defn alias-from-string
  "converts a value from string
 
   (alias-from-string :ref :wallet \"w0\")
   => :wallet.id/w0
 
   (alias-from-string :enum :wallet \"trade\")
   => :trade"
  {:added "3.0"}
  ([type table value]
   (case type
     :ref  (keyword (str (name table) ".id") value)
     :enum (keyword value)
     :string value
     (throw (ex-info "TODO" {:type type :table table :value value})))))

(defn alias-input-single
  "converts an alias to a given key value pair
 
   (alias-input-single {:alias {:wallet-access {:id {:keys [:wallet :account]}}}
                        :schema -schema-}
 
                       :wallet-access :id \"w0.acc0\")
   => {:wallet :wallet-access.id/w0, :account :wallet-access.id/acc0}"
  {:added "3.0"}
  ([{:keys [alias schema] :as store} table k value]
   (let [{:keys [type keys sep]} (get-in alias [table k])
         idx-fn (fn [^String v ^String sep] (.indexOf v sep))
         idx (idx-fn value (or sep "."))]
     (->> (map (fn [k v]
                 (let [[{:keys [type] :as attr}] (get-in schema [:tree table k])]
                   [k (alias-from-string type table v)]))
               keys
               [(subs value 0 idx) (subs value (inc idx))])
          (into {})))))

(defn alias-output
  "constructs an id from `refs`
 
   (alias-output {:alias {:wallet-access {:id {:keys [:wallet :account]}}}
                  :schema -schema-}
                 :wallet-access
                 {:wallet :wallet.id/w0 :account :account.id/acc0})
   => {:wallet :wallet.id/w0, :account :account.id/acc0, :id \"w0.acc0\"}"
  {:added "3.0"}
  ([{:keys [alias schema] :as store} table data]
   (if-let [aliases (get alias table)]
     (reduce-kv (fn [out k {:keys [keys sep]}]
                  (assoc out k (->> (map (fn [k]
                                           (let [[{:keys [type] :as attr}] (get-in schema [:tree table k])]
                                             (alias-to-string type (data k))))
                                         keys)
                                    (string/join (or sep ".")))))
                data
                aliases)
     data)))

(defn input
  "coerces the input into sql format
 
   (input {:schema -schema-}
          :account
          {:id \"account-0\" :wallet :wallet.id/w0})
   => {:id \"account-0\", :wallet-id \"w0\"}"
  {:added "3.0"}
  [store table data]
  (cond (or (nil? data) *raw*) data

        :else
        (->> data
             (ref-input store table)
             (format-input store table))))

(defn output
  "coerces the output from sql format
 
   (output {:schema -schema-}
           :account
           {:id \"account-0\" :wallet-id \"w0\"})
   => {:id \"account-0\", :wallet :wallet.id/w0}"
  {:added "3.0"}
  ([store table data]
   (cond (or (nil? data) *raw*) data

         :else
         (->> data
              (format-output store table)
              (ref-output store table)
              (alias-output store table)))))
