(ns hara.platform.table.atom
  (:require [hara.protocol.table :as protocol.table]
            [hara.core.base.util :as util]))

(defn to-keyword
  "converts an object to a keyword
 
   (to-keyword :hello)
   => :hello
 
   (to-keyword \"hello\")
   => :hello"
  {:added "1.0"}
  [obj]
  (cond (keyword? obj) obj
        (string? obj) (keyword obj)
        (nil? obj) (throw (ex-info "Cannot take nils" {:input obj}))
        :else (keyword (str obj))))

(defn put-atom
  "sets value for the store
 
   (-> (atom {})
       (put-atom :account {:id \"id\"
                           :name \"hello\"})
       (table/get :account \"id\"))
   => {:id \"id\", :name \"hello\"}"
  {:added "1.0"}
  ([atom table {:keys [id] :as m}]
   (doto atom (swap! assoc-in [(to-keyword table)
                               (to-keyword id)]
                     m))))

(defn put-bulk-atom
  "sets multiple values for the store
 
   (-> (atom {})
       (put-bulk-atom {:account [{:id \"id\"
                                  :name \"hello\"}]})
       (table/get :account \"id\"))
   => {:id \"id\", :name \"hello\"}"
  {:added "1.0"}
  ([atom bulk]
   (let [m (reduce-kv (fn [out k arr]
                        (assoc out (to-keyword k) (zipmap (map (comp to-keyword :id) arr)
                                                          arr)))
                      {}
                      bulk)]
     (doto atom (swap! #(merge-with merge % m))))))

(defn set-atom
  "sets the initial value for the store
 
   (-> (atom {})
       (set-atom :account {:id \"id\"
                           :name \"hello\"})
       (set-atom :account {:id \"id\"
                           :name \"world\"}))
   => (throws)"
  {:added "1.0"}
  ([atom table {:keys [id] :as m}]
   (doto atom (swap! update-in [(to-keyword table)
                                (to-keyword id)]
                     (fn [data]
                       (if data (throw (ex-info "Error setting initial value" {:input m})))
                       m)))))

(defn cas-atom
  "compares and swaps value for the store
 
   (-> (atom {})
       (put-atom :account {:id \"id\"
                           :name \"hello\"})
       (cas-atom :account
                 {:id \"id\"
                  :name \"hello\"}
                 {:id \"id\"
                  :name \"world\"})
      (table/get :account \"id\"))
   => {:id \"id\", :name \"world\"}"
  {:added "1.0"}
  ([atom table old {:keys [id] :as new}]
   (cond (not= (:id old) id)
         (throw (ex-info "Id not the same" {:old old :new new}))
         
         :else
         (doto atom (swap! update-in [(to-keyword table)
                                      (to-keyword id)]
                           (fn [m]
                             (if (= m old)
                               new
                               (throw (ex-info "Not updated" {:old old :new new :compare m})))))))))

(defn get-atom
  "gets the value given table and id
 
   (-> (atom {})
       (put-atom :account {:id \"id\"
                           :name \"hello\"})
       (get-atom :account \"id\"))
   => {:id \"id\", :name \"hello\"}"
  {:added "1.0"}
  [atom table id]
  (get-in (deref atom) [(to-keyword table) (to-keyword id)]))

(defn get-in-atom
  "gets value given table and path
   (-> (atom {})
       (put-bulk-atom {:account [{:id \"id\"
                                  :name \"hello\"
                                  :wallet :wallet.id/w0}]
                       :wallet  [{:id \"w0\"
                                  :name \"w0\"}]})
       (get-in-atom :account [\"id\" :wallet]))
   => {:id \"w0\", :name \"w0\"}"
  {:added "1.0"}
  [atom table ids]
  (let [m (deref atom)]
    (reduce (fn [e id]
             (let [val (get e id)]
               (cond (nil? e) (reduced e) 
                     
                     (keyword? val)
                     (let [ns (namespace val)]
                       (if (.endsWith ns ".id")
                         (get-atom atom (subs ns 0 (- (count ns) 3)) (name val))
                         val))
                     
                     :else val)))
            (get-atom atom table (first ids))
            (rest ids))))

(defn delete-atom
  "deletes entry in the store
   (-> (atom {})
       (put-atom :account {:id \"id\"
                           :name \"hello\"})
       
       (delete-atom :account \"id\")
       (table/get :account \"id\"))
   => nil"
  {:added "1.0"}
  ([atom table id]
   (doto atom
     (swap! update (to-keyword table) dissoc (to-keyword id)))))

(defn update-atom
  "updates a key in the store
 
   (-> (atom {})
       (put-atom :account {:id \"id\"
                           :name \"hello\"})
       (update-atom :account \"id\" update [:name str \" world\"])
       (table/get :account \"id\"))
   => {:id \"id\", :name \"hello world\"}"
  {:added "1.0"}
  [atom table id f args]
  (apply swap! atom update-in [(to-keyword table)
                               (to-keyword id)] f args)
  atom)

(defn clear-atom
  "clears all keys in the store
 
   (-> (atom {})
       (put-atom :account {:id \"id\"
                           :name \"hello\"})
       
       (clear-atom)
       (table/get :account \"id\"))
   => nil"
  {:added "1.0"}
  ([atom]
   (doto atom (reset! {})))
  ([atom table]
   (doto atom
     (swap! dissoc (to-keyword table)))))

(defn keys-atom
  "gets all keys in the store
 
   (-> (atom {})
       (put-bulk-atom {:account [{:id \"id\"
                                  :name \"hello\"
                                  :wallet :wallet.id/w0}]
                       :wallet  [{:id \"w0\"
                                  :name \"w0\"}]})
       (keys-atom)
       set)
  => #{:wallet.id/w0 :account.id/id}"
  {:added "1.0"}
  ([atom]
   (mapcat #(keys-atom atom %) (keys @atom)))
  ([atom table]
   (let [m (get @atom (to-keyword table))]
     (map (fn [k]
            (keyword (str (util/keystring table) ".id")
                     (util/keystring k)))
          (keys m)))))

(defn select-atom
  "selects all valid entries given query
 
   (-> (atom {})
       (put-bulk-atom {:account [{:id \"i0\"
                                  :name \"hello\"}
                                 {:id \"i1\"
                                  :name \"world\"}
                                 {:id \"i2\"
                                  :name \"hello\"}]})
       (select-atom :account {:name \"hello\"}))
  => [{:id \"i0\", :name \"hello\"} {:id \"i2\", :name \"hello\"}]"
  {:added "1.0"}
  ([atom table query]
   (let [m (get @atom (to-keyword table))]
     (->> m
          (filter (fn [[id entry]]
                    (every? (fn [[k v]] (if-let [attr (get entry k)]
                                          (= attr v)))
                            query)))
          (map second)))))

(defn list-atom
  "lists all entries in a table
 
   (-> (atom {})
       (put-bulk-atom {:account [{:id \"i0\"
                                  :name \"hello\"}
                                 {:id \"i1\"
                                  :name \"world\"}
                                 {:id \"i2\"
                                  :name \"hello\"}]})
       (list-atom :account))
  => [{:id \"i0\", :name \"hello\"} {:id \"i1\", :name \"world\"} {:id \"i2\", :name \"hello\"}]"
  {:added "1.0"}
  ([atom table]
   (let [m (get @atom (to-keyword table))]
     (map second m))))

(defn bulk-atom
  "updates entries in the atom
 
   (-> (atom {})
       (bulk-atom {:put {:account [{:id \"id\"
                                    :name \"hello\"
                                    :wallet :wallet.id/w0}]
                         :wallet  [{:id \"w0\"
                                    :name \"w0\"}]}})
       (bulk-atom {:cas {:account [[{:id \"id\"
                                     :name \"hello\"
                                    :wallet :wallet.id/w0}
                                    {:id \"id\"
                                     :name \"world\"
                                     :wallet :wallet.id/w0}]]}})
       (list-atom :account))
   => [{:id \"id\", :name \"world\", :wallet :wallet.id/w0}]"
  {:added "1.0"}
  ([atom {:keys [put set cas delete] :as input}]
   (doto atom
     (swap! (fn [m]
              (let [out-data (reduce-kv (fn [out table arr]
                                          (assoc out (to-keyword table)
                                                 (zipmap (map (comp to-keyword :id) arr)
                                                         arr)))
                                        {}
                                        (:put input))
                    put-data (merge-with merge m out-data)
                    set-data (reduce-kv (fn [out table arr]
                                          (assoc out (to-keyword table)
                                                 (mapv (fn [v] [nil v]) arr)))
                                        {}
                                        (:set input))
                    cas-data (reduce-kv (fn [out table arr]
                                          (reduce (fn [out [old new]]
                                                    (let [_ (if-not (or (nil? old)
                                                                        (= (:id old) (:id new)))
                                                              (throw (ex-info "Id not the same" {:old old :new new})))]
                                                      (update-in out [(to-keyword table)
                                                                      (to-keyword (:id new))]
                                                                 (fn [v]
                                                                   (if (= v old)
                                                                     new
                                                                     (throw (ex-info "Not updated" {:old old :new new :compare v})))))))
                                                  out
                                                  arr))
                                        put-data
                                        (merge-with concat set-data (:cas input)))
                    delete-data (reduce-kv (fn [out table ids]
                                             (reduce (fn [out id]
                                                       (update out (to-keyword table) dissoc (to-keyword id)))
                                                     out
                                                     ids))
                                           cas-data
                                           (:delete input))]
                delete-data))))))

(extend-type clojure.lang.Atom
  protocol.table/ITable
  (protocol.table/-put
    ([atom table m opts] (put-atom atom table m))
    ([atom m] (put-bulk-atom atom m)))
  (protocol.table/-set
    ([atom table m opts] (set-atom atom table m)))
  (protocol.table/-cas
    ([atom table old new opts] (cas-atom atom table old new)))
  (protocol.table/-delete
    ([atom table id opts]
     (delete-atom atom table id)))
  (protocol.table/-update
    ([atom table id f args opts]
     (update-atom atom table id f args)))
  (protocol.table/-clear
    ([atom] (clear-atom atom))
    ([atom table opts] (clear-atom atom table)))
  (protocol.table/-bulk
    ([atom m opts] (bulk-atom atom m)))
  (protocol.table/-db
    ([atom] atom)
    ([atom time opts] atom))

  protocol.table/IQuery
  (protocol.table/-get
    ([atom table id opts]
     (get-atom atom table id)))
  (protocol.table/-get-in
    ([atom table ids opts]
     (get-in-atom atom table ids)))
  (protocol.table/-keys 
    ([atom] (keys-atom atom))
    ([atom table opts] (keys-atom atom table)))
  (protocol.table/-select
    ([atom table opts]
     (list-atom atom table))
    ([atom table query opts]
     (select-atom atom table query))))

(defmethod protocol.table/-create-datastore :atom
  [m]
  (atom {}))
