(ns hara.platform.table
  (:require [hara.protocol.table :as protocol.table]
            [hara.platform.table.atom :as atom]
            [hara.platform.table.base :as base]
            [hara.core.base.util :as util]
            [hara.data.schema :as schema]
            [hara.data.pipeline :as pipeline]
            [hara.core.component :as component])
  (:refer-clojure :exclude [get set count update keys get-in ref list flatten]))

(defn table?
  "checks if the store is of type table
 
   (table? (atom {}))
   => true"
  {:added "1.0"}
  ([obj]
   (satisfies? protocol.table/ITable obj)))

(defn ref
  "constructs a reference
 
   (table/ref :account \"hello\")
   => :account.id/hello"
  {:added "1.0"}
  ([table id]
   (keyword (str (util/keystring table) ".id") (util/keystring id))))

(defn put
  "puts key-value pairs of the datastore"
  {:added "1.0"}
  ([store m]
   (protocol.table/-put store m))
  ([store table m]
   (protocol.table/-put store table m nil))
  ([store table m opts]
   (protocol.table/-put store table m opts)))

(defn set
  "sets initial key-value pair of the datastore"
  {:added "1.0"}
  ([store table m]
   (protocol.table/-set store table m nil))
  ([store table m opts]
   (protocol.table/-set store table m opts)))

(defn cas
  "compares and swaps two values in the database"
  {:added "1.0"}
  ([store table old new]
   (protocol.table/-cas store table old new nil))
  ([store table old new opts]
   (protocol.table/-cas store table old new opts)))

(defn get
  "gets the value for a keys"
  {:added "1.0"}
  ([store table id]
   (protocol.table/-get store table id nil))
  ([store table id opts]
   (protocol.table/-get store table id opts)))

(defn get-in
  "walks keys on the datastore"
  {:added "1.0"}
  ([store table ids]
   (protocol.table/-get-in store table ids nil))
  ([store table ids opts]
   (protocol.table/-get-in store table ids opts)))

(defn delete
  "deletes the key on the datastore"
  {:added "1.0"}
  ([datastore table id]
   (protocol.table/-delete datastore table id nil))
  ([datastore table id opts]
   (protocol.table/-delete datastore table id opts)))

(defn update
  "updates a key on the datastore"
  {:added "1.0"}
  ([datastore table id f args]
   (protocol.table/-update datastore table id f args nil))
  ([datastore table id f args opts]
   (protocol.table/-update datastore table id f args opts)))

(defn clear
  "clears all or only keys containing a given attribute"
  {:added "1.0"}
  ([datastore]
   (protocol.table/-clear datastore))
  ([datastore table opts]
   (protocol.table/-clear datastore table opts)))

(defn keys
  "returns all the keys of the atom"
  {:added "1.0"}
  ([store]
   (protocol.table/-keys store))
  ([store table opts]
   (protocol.table/-keys store table opts)))

(defn count
  "returns all the keys in the table
   
   (-> (atom {})
       (table/put {:account [{:id \"id\"
                              :name \"hello\"
                              :wallet :wallet.id/w0}]
                  :wallet  [{:id \"w0\"
                              :name \"w0\"}]})
       (table/count :account))"
  {:added "3.0"}
  ([store table]
   (protocol.table/-count store table {} {}))
  ([store table query]
   (protocol.table/-count store table query {}))
  ([store table query opts]
   (protocol.table/-count store table query opts)))

(defn select
  "returns all entries with attributes"
  {:added "1.0"}
  ([store table]
   (protocol.table/-select store table {} {}))
  ([store table query]
   (protocol.table/-select store table query {}))
  ([store table query opts]
   (protocol.table/-select store table query opts)))

(defn bulk
  "updates entries in the atom using :put and :cas"
  {:added "1.0"}
  ([store m]
   (protocol.table/-bulk store m nil))
  ([store m opts]
   (protocol.table/-bulk store m opts)))

(defn install-schema
  "installs the schema for the store
 
   (doto (table/datastore {:type :mock})
     (table/install-schema (schema/schema [:user [:id {:type :string}]])))
   => (throws)"
  {:added "1.0"}
  ([store schema]
   (protocol.table/-install-schema store schema)))
  
(defn uninstall-schema
  "installs the schema for the store
 
   (doto (table/datastore {:type :mock})
     (table/uninstall-schema (schema/schema [:user [:id {:type :string}]])))
   => (throws)"
  {:added "1.0"}
  ([store schema]
   (protocol.table/-uninstall-schema store schema)))

(defn db
  "returns the query store
 
   (table/db (atom {}))
   => clojure.lang.Atom
   
   (table/db (table/datastore {}))
   => hara.platform.table.base.MockStore"
  {:added "1.0"}
  ([store]
   (protocol.table/-db store))
  ([store time opts]
   (protocol.table/-db store time opts)))

(defn create
  "constructs a datastore
 
   (table/create {:type :mock})
   => hara.platform.table.base.MockStore"
  {:added "1.0"}
  ([] (create {}))
  ([{:keys [type data-dir]
     :or {type :mock} :as opts}]
   (protocol.table/-create-datastore (merge opts {:type type}))))
   
(defn datastore
  "constructs a datastore
 
   (table/datastore {:type :mock})
   => hara.platform.table.base.MockStore"
  {:added "1.0"}
  ([]
   (datastore {:type :mock}))
  ([m]
   (-> (create m)
       (component/start))))
