(ns hara.platform.storage.file
  (:require [hara.protocol.storage :as protocol.storage]
            [hara.protocol.component :as protocol.component]
            [hara.io.file :as fs]))

(defn write-file
  "writes bytes to storage
 
   (write-file -storage- \"hello.txt\" (.getBytes \"Hello World\"))"
  {:added "0.1"}
  ([{:keys [root]} key bytes]
   (let [path   (fs/path root key)
         parent (fs/parent path)]
     (if-not (fs/exists? parent) (fs/create-directory parent))
     (fs/write-all-bytes path bytes))))

(defn read-file
  "reads bytes from storage
   
   (-> (read-file -storage- \"hello.txt\")
       (String.))
   => \"Hello World\""
  {:added "0.1"}
  ([{:keys [root]} key]
   (let [path (fs/path root key)]
     (if (fs/exists? path)
       (fs/read-all-bytes path)))))

(defn list-file
  "lists all files in the storage
 
   (list-file -storage-)
   => [\"hello.txt\"]"
  {:added "0.1"}
  ([{:keys [root]}]
   (let [root (fs/path root)]
     (->> (fs/list root {:recursive true
                         :exclude [fs/directory?]})
          (map (comp str #(fs/relativize root %) first))))))

(defn info-file
  "look at file attribute within the storage
 
   (info-file -storage- \"hello.txt\")
   => (contains {:time number?, :size 11})"
  {:added "0.1"}
  ([{:keys [root]} key]
   (let [path (fs/path root key)]
     (if (fs/exists? path)
       (let [{time :last-modified-time size :size} (fs/attributes path)]
         {:time time :size size})))))

(defn clear-file
  "clears all files in the storage
 
   (clear-file -storage- \"hello.txt\")
   
   (clear-file -storage-)"
  {:added "0.1"}
  ([{:keys [root]}]
   (fs/delete root)
   (fs/create-directory root))
  ([{:keys [root]} key]
   (fs/delete (fs/path root key))))

(defrecord FileStorage [root]
  Object
  (toString [storage]
    (str {:items (count (list-file storage))}))
  
  protocol.storage/IStorage
  (-write    [storage key bytes] (write-file storage key bytes) storage)
  (-read     [storage key] (read-file storage key))
  (-list     [storage] (list-file storage))
  (-info     [storage key] (info-file storage key))
  (-clear    [storage] (clear-file storage) storage)
  (-clear    [storage key] (clear-file storage key) storage)

  protocol.component/IComponent
  (-start [{:keys [initial] :as storage}]
    (doseq [[k bytes] initial]
      (write-file storage k bytes))
    storage)
  (-stop [storage] storage))

(defmethod print-method FileStorage
  ([v ^java.io.Writer w]
   (.write w (str #"storage.file " v))))

(defmethod protocol.storage/-create :file
  ([m]
   (map->FileStorage m)))

(comment
  (use 'hara.tool)
  (./scaffold)
  
  )
