(ns hara.platform.mail.interop.flag
  (:require [clojure.set :as set]
            [hara.object :as object]
            [hara.object.query :as reflect])
  (:import [javax.mail Flags Flags$Flag]))

(def construct (reflect/query-class Flags$Flag ["new" :#]))

(def bit (reflect/query-class Flags$Flag ["bit" :#]))

(def lookup
  (->> (drop-last (reflect/query-class Flags$Flag [Flags$Flag]))
       (map (juxt  #(bit (% Flags$Flag)) :name))
       (into {})))

(def rlookup
  (reduce-kv (fn [out k v] (assoc out v k)) {} lookup))

(object/string-like
 Flags$Flag
 {:tag "flag"
  :read  (comp lookup bit)
  :write (comp construct rlookup)})

(defn read-flags
  "reads all flags as strings
 
   (read-flags (doto (Flags.)
                 (.add Flags$Flag/ANSWERED)
                 (.add Flags$Flag/DELETED)
                 (.add \"CUSTOM-00\")
                 (.add \"CUSTOM-01\")))
   => [\"ANSWERED\" \"CUSTOM-00\" \"CUSTOM-01\" \"DELETED\"]"
  {:added "3.0"}
  [^Flags flags]
  (let [sys  (.getSystemFlags flags)
        user (.getUserFlags flags)]
    (vec (sort (concat (map object/to-data sys)
                       (seq user))))))

(defn write-flags
  "writes strings as a Flag
 
   (def ^Flags flags (write-flags [\"ANSWERED\" \"CUSTOM-00\" \"CUSTOM-01\" \"DELETED\"]))
   
   "
  {:added "3.0"}
  [arr]
  (let [all  (set (keys rlookup))
        arr  (set arr)
        sys  (set/intersection all arr)
        user (set/difference arr all)
        flags (Flags.)]
    (doseq [s user]
      (.add flags ^String s))
    (doseq [s sys]
      (.add flags ^Flags$Flag (object/from-data s Flags$Flag)))
    flags))

(object/vector-like
 Flags
 {:tag "flags"
  :read read-flags
  :write write-flags})
