(ns hara.log.core
  (:require [hara.protocol.log :as protocol.log]
            [hara.protocol.component :as protocol.component]
            [hara.io.concurrent :as cc]
            [hara.log.common :as common]
            [hara.log.match :as match])
  (:import (java.text SimpleDateFormat)))

(defn logger-enqueue-core
  "submits a task to the logger job queue"
  {:added "3.0"}
  [{:keys [instance] :as logger} entry]
  (let [{:keys [queue executor interval max-batch]} @instance]
    (cc/add queue entry)
    (cc/submit-notify executor
                      (fn []
                        (cc/process-queue (fn [entries]
                                            (protocol.log/-logger-process logger entries))
                                          queue
                                          max-batch))
                      interval)
    :logger/enqueued))

(defn logger-write-core
  "writes a entry to the logger"
  {:added "3.0"}
  [{:keys [instance] :as logger} entry]
  (let [{:keys [context] :as params} @instance]
    (if (match/match params entry)
      (logger-enqueue-core logger (merge context entry))
      :logger/ignored)))

(defn logger-info-core
  "returns information about the logger
 
   (logger-info-core log/*logger*)
   => {:interval 200, :items 0}"
  {:added "3.0"}
  ([{:keys [instance] :as logger}]
   (let [{:keys [queue] :as m} @instance]
     (-> (dissoc m :executor)
         (assoc :queue (count queue))))))

(defn logger-start-core
  "starts the logger, initating a queue and executor"
  {:added "3.0"}
  [{:keys [instance] :as logger}]
  (let [queue    (cc/queue)
        executor (cc/single-executor 1)]
    (swap! instance merge {:queue queue
                           :executor executor})
    logger))

(defn logger-stop-core
  "stops the logger, queue and executor"
  {:added "3.0"}
  [{:keys [instance] :as logger}]
    (cc/shutdown (:executor @instance))
    (swap! instance dissoc :queue :executor)
  logger)

(defn logger-init-core
  "sets defaults for the logger
 
   (logger-init-core {})
   => (contains-in {:instance clojure.lang.Atom,
                    :level :info, :interval 200, :max-batch 1000})"
  {:added "3.0"}
  [{:keys [type] :as m}]
  {:type type
   :instance (atom (merge {:level :info}
                          (dissoc m :type)))})

(defrecord IdentityLogger [instance]
  
  Object
  (toString [logger] (str "#log.identity" (logger-info-core logger)))

  protocol.log/ILogger
  (-logger-write [{:keys [instance]} entry]
    (if (match/match @instance entry)
      entry)))

(defmethod print-method IdentityLogger
  [v ^java.io.Writer w]
  (.write w (str v)))

(defmethod protocol.log/-create :identity
  [m]
  (-> (logger-init-core m)
      (map->IdentityLogger)))

(defn identity-logger
  "creates a identity logger
 
   (identity-logger)"
  {:added "3.0"}
  ([] (identity-logger nil))
  ([m]
   (protocol.log/-create (assoc m :type :identity))))

(defrecord MultiLogger [instance loggers]
  
  Object
  (toString [_] (str "#log.multi" (assoc @instance :loggers loggers)))
  
  protocol.log/ILogger
  (-logger-write [{:keys [instance]} entry]
    (doseq [logger @instance]
      (protocol.log/-logger-write logger entry)))

  protocol.component/IComponent
  (-start [logger]
    (update logger :loggers #(map protocol.component/-start %)))

  (-stop [{:keys [instance] :as logger}]
    (update logger :loggers #(map protocol.component/-stop %))))

(defmethod print-method MultiLogger
  [v ^java.io.Writer w]
  (.write w (str v)))

(defmethod protocol.log/-create :multi
  [{:keys [type loggers] :as m}]
  (-> {:type type
       :instance (atom (dissoc m :type :loggers))
       :loggers (mapv protocol.log/-create loggers)}
      (map->MultiLogger)))

(defn multi-logger
  "creates multiple loggers
 
   (multi-logger {:loggers [{:type :console
                             :interval 500
                            :max-batch 10000}
                            {:type :basic
                             :interval 500
                             :max-batch 10000}]})"
  {:added "3.0"}
  ([m]
   (-> (protocol.log/-create (assoc m :type :multi))
       (protocol.component/-start))))

(defn log-write
  "writes a message to the logger"
  {:added "3.0"}
  ([logger level meta message data]
   (log-write logger level meta message data nil))
  ([logger level meta message data timestamp]
   (let [data (if (map? data)
                data
                {:log/value data})]
     (protocol.log/-logger-write logger
                                 (-> (assoc meta
                                            :log/level level
                                            :log/message message
                                            :log/timestamp (or timestamp (System/currentTimeMillis)))
                                     (merge common/*context* data))))))
