(ns hara.platform.exchange.log
  (:require [hara.protocol.exchange :as protocol.exchange]
            [hara.protocol.log :as protocol.log]
            [hara.protocol.component :as protocol.component]
            [hara.core.component :as component]
            [hara.log.core :as core]
            [hara.io.concurrent :as cc]))

(defn logger-process-exchange
  "process the exchange logger"
  {:added "3.0"}
  ([{:keys [exchange instance] :as logger} items]
   (let [{:keys [publish]} @instance]
     (doseq [item items]
       (protocol.exchange/-publish exchange publish (prn-str item) {})))))

(defrecord ExchangeLogger [instance]
  
  Object
  (toString [logger] (str "#log.exchange" (core/logger-info-core logger)))

  protocol.log/ILogger
  (-logger-write [logger entry] (core/logger-write-core logger entry))
  
  protocol.log/ILoggerProcess
  (-logger-process [logger items] (logger-process-exchange logger items))

  protocol.component/IComponent
  (-start [{:keys [connect] :as logger}]
    (let [exchange (if connect
                     (-> (protocol.exchange/-create connect)
                         (component/start)))]
      (cond-> logger
        exchange (assoc :exchange exchange))))

  (-stop [{:keys [connect exchange] :as logger}]
    (let [_  (if connect
               (component/stop exchange))]
      (cond-> logger
        connect (dissoc :exchange)))))

(defmethod print-method ExchangeLogger
  [v ^java.io.Writer w]
  (.write w (str v)))

(defmethod protocol.log/-create :exchange
  [m]
  (let [queue (cc/queue)]
    (-> (merge {:interval 200
                :max-batch 1000
                :queue queue
                :executor (cc/single-executor 1)}
               m)
        (core/logger-init-core)   
        (map->ExchangeLogger))))

(defn exchange-logger
  "creates a exchange logger
 
   (exchange-logger {:interval 500 :max-batch 10000})"
  {:added "3.0"}
  ([] (exchange-logger nil))
  ([m]
   (-> (protocol.log/-create (assoc m :type :exchange))
       (component/start))))

