(ns hara.platform.cache.publisher
  (:require [hara.protocol.cache :as protocol.cache]
            [hara.platform.cache.base :as base]
            [hara.protocol.component :as protocol.component]
            [hara.core.component :as component]))

(defn start-publisher
  [{:keys [notify connect instance] :as cache}]
  (let [remote (-> (protocol.cache/-create connect)
                   (component/start))
        local  (-> (protocol.cache/-create {:type :mock
                                            :sync :write
                                            :source remote
                                            :notify notify})
                   (component/start))]
    (swap! instance assoc :local local :remote remote)
    cache))

(defn stop-publisher
  [{:keys [instance] :as cache}]
  (let [{:keys [local remote]} @instance]
    (component/stop local)
    (component/stop remote)
    (swap! instance dissoc :local :remote)
    cache))

(defrecord PublisherCache [instance]

  Object
  (toString [{:keys [show] :as cache}]
    (let [{:keys [local remote]} @instance]
      (str "#cache.publisher"
           (if local
             (case show
               :keys  (vec (protocol.cache/-keys local))
               :all  (protocol.cache/-all local)
               [(protocol.cache/-count local)])
             "<uninitiased>"))))

  protocol.cache/ICache
  (-all    [cache] (protocol.cache/-all (:local @instance)))
  (-count  [cache] (protocol.cache/-count (:local @instance)))
  (-expired? [cache key] (protocol.cache/-expired? (:local @instance) key))
  (-expiry [cache key] (protocol.cache/-expiry (:local @instance) key))
  (-get    [cache key]
    (or (protocol.cache/-get (:local @instance) key)
        (protocol.cache/-get (:local @instance) key)))
  (-has?   [cache key] (protocol.cache/-has? (:local @instance) key))
  (-keys   [cache] (protocol.cache/-keys (:local @instance)))
  (-keys   [cache pattern] (protocol.cache/-keys (:local @instance) pattern))

  (-batch [cache add-values add-expiry remove-vec]
    (protocol.cache/-batch (:local @instance) add-values add-expiry remove-vec)
    cache)
  (-clear  [cache]
    (protocol.cache/-clear (:local @instance))
    (protocol.cache/-clear (:local @instance))
    cache)
  (-delete [cache keys]
    (protocol.cache/-delete (:local @instance) keys)
    cache)
  (-set    [cache key value]
    (protocol.cache/-set (:local @instance) key value)
    cache)
  (-set [cache key value expiry]
    (protocol.cache/-set (:local @instance) key value expiry)
    cache)
  (-touch [cache key expiry]
    (protocol.cache/-touch (:local @instance) key expiry)
    cache)
  (-export [cache key]
    (protocol.cache/-get (:local @instance) key))
  (-import [cache key value]
    (if value
      (protocol.cache/-set  (:local @instance) key value)
      (protocol.cache/-delete (:local @instance) [key])))
  
  protocol.cache/ICacheExtend
  (-mget     [cache keys] (protocol.cache/-mget (:local @instance) keys))
  (-mset     [cache args] (protocol.cache/-mset (:local @instance) args))
  (-bulk     [cache args] (protocol.cache/-bulk (:local @instance) args))
  (-transact [cache args] (protocol.cache/-transact (:local @instance) args))
  (-notify   [cache name pattern f] (protocol.cache/-notify (:local @instance) name pattern f))
  (-denotify [cache name] (protocol.cache/-denotify (:local @instance) name))

  protocol.component/IComponent
  (-start [cache]
    (start-publisher cache))

  (-stop [cache]
    (stop-publisher cache)))

(defmethod print-method PublisherCache
  [v ^java.io.Writer w]
  (.write w (str v)))

(defmethod protocol.cache/-create :publisher
  [{:keys [notify] :as m}]
  (map->PublisherCache (assoc m :instance (atom {}))))
