(ns hara.platform.cache.base
  (:require [hara.protocol.cache :as protocol.cache]
            [hara.protocol.component :as protocol.component]
            [hara.platform.cache.atom :as atom]
            [hara.print.table :as table]
            [hara.core.base.util :as util]
            [hara.data.base.map :as map]))

(defn pattern-regex
  "transforms a pattern string to regex
 
   (pattern-regex \"*\")
   => #\".*\""
  {:added "3.0"}
  [^String pattern]
  (let [pattern (-> pattern
                    (.replaceAll "\\*" ".*")
                    (.replaceAll "\\?" "."))
        regex (re-pattern pattern)]
    regex))

(defn notify-key
  "call function handlers with key
 
   (def -out- (atom nil))
   
   (doto (cache/cache {:type :mock})
     (notify-cache :all \"*\" (fn [_ key] (reset! -out- key)))
     (notify-key \"a\"))
 
   @-out- => \"a\""
  {:added "3.0"}
  [{:keys [instance] :as cache} key]
  (let [{:keys [notifications]} @instance]
    (doseq [[id {:keys [pattern handler]}] notifications]
      (if (re-find (pattern-regex pattern) (util/keystring key))
        (handler cache key)))))

(defn wrap-notify
  "runs notifications when called
 
   (def -out- (atom nil))
   
   (doto (cache/cache {:type :mock})
     (notify-cache :all \"*\" (fn [_ key] (reset! -out- key)))
     (cache/set :a \"hello\"))
 
   @-out- => :a
 
   (def -out- (atom nil))
   (doto (cache/cache {:type :mock})
     (notify-cache :all \"b\" (fn [_ key] (reset! -out- key)))
     (cache/set :a \"hello\"))
 
   @-out- => nil"
  {:added "3.0"}
  [f cache]
  (fn [state key & args]
    (let [output (apply f state key args)]
      (notify-key cache key)
      output)))

(defn wrap-notify-delete
  {:added "3.0"}
  [f {:keys [instance] :as cache}]
  (fn [state keys]
    (let [output (f state keys)
          _    (doseq [k keys]
                 (notify-key cache k))]
      output)))

(defn wrap-notify-bulk
  "notifications for bulk operations
 
   (def -out- (atom #{}))
   
   (doto (cache/cache {:type :mock})
     (notify-cache :all \"*\" (fn [_ key] (swap! -out- conj key)))
     (cache/mset :a \"hello\" :b \"hello\" :c \"hello\"))
   
   @-out- => #{:c :b :a}"
  {:added "3.0"}
  [f {:keys [instance] :as cache}]
  (fn [state args]
    (let [count  (count args)
          output (f state args)
          ks   (if (= count 1)
                 (keys (first args))
                 (map first (partition 2 args)))
          _    (doseq [k ks]
                 (notify-key cache k))]
      output)))

(defn notify-cache
  "adds a listener to the cache"
  {:added "3.0"}
  [{:keys [instance] :as cache} id pattern f]
  (let [{:keys [notifications]} @instance
        _  (if (get notifications id)
             (throw (ex-info "Already registered" {:id id})))
        _ (swap! instance assoc-in [:notifications id] {:pattern pattern
                                                        :handler f})]
    cache))

(defn denotify-cache
  "removes a listener from the cache"
  {:added "3.0"}
  [{:keys [instance] :as cache} id]
  (let [{:keys [notifications]} @instance
        {:keys [pattern]} (or (get notifications id)
                              (ex-info "Not registered" {:id id}))
        _  (swap! instance update :notifications dissoc id)]
    cache))

(defn init-notifications
  "TODO"
  {:added "3.0"}
  [{:keys [instance] :as cache}]
  (doseq [[id {:keys [pattern handler]}] (:notify cache)]
    (notify-cache cache id pattern handler)))

(defn read-sync
  [{:keys [source] :as cache}]
  (let [keys (protocol.cache/-keys source "*")
        vals  (->> (map (partial protocol.cache/-export source) keys)
                   (zipmap keys))]
    ;;(prn vals)
    (protocol.cache/-mset cache [vals])))

(defn write-sync
  [{:keys [source] :as cache}]
  (let [keys (protocol.cache/-keys cache "*")]
    (doseq [key keys]
      (->> (protocol.cache/-export cache key)
           (protocol.cache/-import source key)))))

(defn init-sync
  [{:keys [instance source sync] :as cache}]
  (let [sync-id  (util/sid)
        sync-key (keyword "sync" sync-id)
        _ (swap! instance assoc :sync-key sync-key)
        _ (case sync
            :read  (do (future (read-sync cache))
                       (protocol.cache/-notify source sync-key "*" (fn [_ key]
                                                                     (->> (protocol.cache/-export source key)
                                                                          (protocol.cache/-import cache key)))))
            :write (do (future (protocol.cache/-clear source)
                               (write-sync cache))
                       (notify-cache cache :sync "*" (fn [_ key]
                                                       (->> (protocol.cache/-export cache key)
                                                            (protocol.cache/-import source key))))))]
    cache))

(defn stop-sync
  [{:keys [instance source sync] :as cache}]
  (let [sync-key (get @instance :sync-key)
        _ (swap! instance dissoc :sync-key)
        _ (case sync
            :read  (protocol.cache/-denotify source sync-key)
            :write (denotify-cache cache :sync))]
    cache))

(defrecord MockCache [state]

  Object
  (toString [{:keys [state show] :as cache}]
    (str "#cache.mock"
         (if state
           (case show
             :keys  (vec (protocol.cache/-keys cache))
             :data  (protocol.cache/-all state)
             [(protocol.cache/-count cache)])
           "<uninitiased>")))

  protocol.cache/ICache
  (-all    [cache] (protocol.cache/-all state))
  (-count  [cache] (protocol.cache/-count state))
  (-expired? [cache key] (protocol.cache/-expired? state key))
  (-expiry [cache key] (protocol.cache/-expiry state key))
  (-get    [cache key] (protocol.cache/-get state key))
  (-has?   [cache key] (protocol.cache/-has? state key))
  (-keys   [cache] (protocol.cache/-keys state))
  (-keys   [cache pattern] (protocol.cache/-keys state pattern))

  (-batch [cache add-values add-expiry remove-vec]
    (protocol.cache/-batch state add-values add-expiry remove-vec)
    cache)
  (-clear  [cache]
    (protocol.cache/-clear state)
    cache)
  (-delete [cache keys]
    ((wrap-notify-delete protocol.cache/-delete cache) state keys)
    cache)
  (-set    [cache key value]
    ((wrap-notify protocol.cache/-set cache) state key value)
    cache)
  (-set [cache key value expiry]
    ((wrap-notify protocol.cache/-set cache) state key value expiry)
    cache)
  (-touch [cache key expiry]
    (protocol.cache/-touch state key expiry)
    cache)
  (-export [cache key]
    (protocol.cache/-get cache key))
  (-import [cache key value]
    (if value
      (protocol.cache/-set cache key value)
      (protocol.cache/-delete cache [key])))
  
  protocol.cache/ICacheExtend
  (-mget     [cache keys] (protocol.cache/-mget state keys))
  (-mset     [cache args] ((wrap-notify-bulk protocol.cache/-mset cache) state args))
  (-bulk     [cache args] (protocol.cache/-bulk state args))
  (-transact [cache args] (protocol.cache/-transact state args))
  (-notify   [cache name pattern f] (notify-cache cache name pattern f))
  (-denotify [cache name] (denotify-cache cache name))

  protocol.cache/ICacheList
  (-lindex   [cache key index] (protocol.cache/-lindex state key index))
  (-llen     [cache key] (protocol.cache/-llen state key))
  (-lpop     [cache key] ((wrap-notify protocol.cache/-lpop cache) state key) cache)
  (-lpush    [cache key elements] ((wrap-notify protocol.cache/-lpush cache) state key elements) cache)
  (-lrange   [cache key start stop] (protocol.cache/-lrange state key start stop))
  (-lrem     [cache key count value] ((wrap-notify protocol.cache/-lrem cache) state key count value))
  (-lset     [cache key index element] ((wrap-notify protocol.cache/-lset cache) state key index element) cache)
  (-ltrim    [cache key start stop] ((wrap-notify protocol.cache/-ltrim cache) state key start stop) cache)
  (-rpop     [cache key] ((wrap-notify protocol.cache/-rpop cache) state key) cache)
  (-rpush    [cache key elements] ((wrap-notify protocol.cache/-rpush cache) state key elements) cache)

  protocol.cache/ICacheHash
  (-hdel     [cache key fields] ((wrap-notify protocol.cache/-hdel cache) state key fields) cache)
  (-hexists  [cache key field] (protocol.cache/-hexists state key field))
  (-hget     [cache key field] (protocol.cache/-hget state key field))
  (-hmget    [cache key fields] (protocol.cache/-hmget state key fields))
  (-hgetall  [cache key] (protocol.cache/-hgetall state key))
  (-hkeys    [cache key] (protocol.cache/-hkeys state key))
  (-hlen     [cache key] (protocol.cache/-hlen state key))
  (-hset     [cache key args] ((wrap-notify protocol.cache/-hset cache) state key args) cache)
  (-hvals    [cache key] (protocol.cache/-hvals state key))

  protocol.cache/ICacheSortedSet
  (-zadd     [cache key args] ((wrap-notify protocol.cache/-zadd cache) state key args))
  (-zcard    [cache key] (protocol.cache/-zcard state key))
  (-zcount   [cache key min max] (protocol.cache/-zcount state key min max))
  (-zpopmin  [cache key count] ((wrap-notify protocol.cache/-zpopmin cache) state key count))
  (-zpopmax  [cache key count] ((wrap-notify protocol.cache/-zpopmax cache) state key count))
  (-zrange   [cache key start stop] (protocol.cache/-zrange state key start stop))
  (-zrangebyscore [cache key min max] (protocol.cache/-zrangebyscore state key min max)) 
  (-zrank    [cache key member] (protocol.cache/-zrank state key member))
  (-zrem     [cache key members] ((wrap-notify protocol.cache/-zrem cache) state key members))
  (-zremrangebyscore [cache key min max] ((wrap-notify protocol.cache/-zremrangebyscore cache) state key min max)) 
  (-zrevrange [cache key start stop] (protocol.cache/-zrevrange state key start stop))
  (-zrevrank  [cache key member] (protocol.cache/-zrevrank state key member))
  (-zscore  [atom key member] (protocol.cache/-zscore state key member))

  protocol.component/IComponent
  (-start [{:keys [file log sync source] :as cache}]
    (cond-> (table/attach-state cache)
      (and sync source) (doto (init-sync))))

  (-stop [{:keys [file log sync source] :as cache}]
    (cond-> (table/detach-state cache)
      (and sync source) (doto (stop-sync)))))

(defmethod print-method MockCache
  [v ^java.io.Writer w]
  (.write w (str v)))

(defmethod protocol.cache/-create :mock
  [{:keys [notify] :as m}]
  (let [cache (map->MockCache (assoc m :instance (atom {})))
        _     (init-notifications cache)]
    cache))
