(ns hara.platform.cache
  (:require [hara.protocol.cache :as protocol.cache]
            [hara.core.component :as component]
            [hara.platform.common :as common]
            [hara.platform.cache.base :deps true]
            [hara.platform.cache.receiver :deps true]
            [hara.platform.cache.publisher :deps true]
            [hara.platform.cache.log :deps true])
  (:refer-clojure :exclude [get set keys count]))

(defn set
  "sets the value with or optional expiry
 
   (-> (cache)
       (cache/set :a 1)
       :state
       deref)
   => {:a {:value 1}}"
  {:added "3.0"}
  ([cache key value]
   (protocol.cache/-set cache key value))
  ([cache key value expiry]
   (protocol.cache/-set cache key value expiry)))

(defn get
  "returns the value of a key
   
   (-> (cache {:type :mock
               :initial {:a {:value 1} :b {:value 2} :c {:value 3}}})
       (cache/get :a))
   => 1"
  {:added "3.0"}
  [cache key]
  (protocol.cache/-get cache key))

(defn has?
  "checks if key exists
 
   (-> (cache {:type :mock
               :initial {:a {:value 1} :b {:value 2} :c {:value 3}}})
       (cache/has? :a))
   => true"
  {:added "3.0"}
  [cache key]
  (protocol.cache/-has? cache key))

(defn count
  "returns number of active keys in the cache
 
   (-> (cache {:type :mock
               :initial {:a {:value 1} :b {:value 2}}})
       (cache/count))
   => 2"
  {:added "3.0"}
  [cache]
  (protocol.cache/-count cache))

(defn batch
  "performs multiple operations in the cache
 
   (-> (cache {:type :mock
               :initial {:a {:value 1} :b {:value 2}}})
       (cache/batch {:c 3 :d 4} {} [:b])
       (cache/all))
   => {:a 1, :c 3, :d 4}"
  {:added "3.0"}
  ([cache add-values]
   (protocol.cache/-batch cache add-values {} []))
  ([cache add-values add-expiry remove-vec]
   (protocol.cache/-batch cache add-values add-expiry remove-vec)))

(defn delete
  "deletes given key
 
   (-> (cache {:type :mock
               :initial {:a {:value 1} :b {:value 2}}})
       (cache/delete :b)
       (cache/all))
   => {:a 1}"
  {:added "3.0"}
  ([cache] cache)
  ([cache key]
   (protocol.cache/-delete cache [key]))
  ([cache key & more]
   (protocol.cache/-delete cache (cons key more))))

(defn clear
  "clears the cache
 
   (-> (cache {:type :mock
               :initial {:a {:value 1} :b {:value 2}}})
       (cache/clear)
       (cache/all))
   => {}"
  {:added "3.0"}
  [cache]
  (protocol.cache/-clear cache))

(defn all
  "returns all values in the cache
   
   (-> (cache {:type :mock
               :initial {:a {:value 1} :b {:value 2}}})
       (cache/all))
   => {:a 1, :b 2}"
  {:added "3.0"}
  [cache]
  (protocol.cache/-all cache))

(defn keys
  "returns all keys in the cache
 
   (-> (cache {:type :mock
               :initial {:a {:value 1} :b {:value 2}}})
       (cache/keys)
       sort)
   => [:a :b]"
  {:added "3.0"}
  ([cache]
   (protocol.cache/-keys cache))
  ([cache match]
   (protocol.cache/-keys cache match)))

(defn touch
  "renews the expiration time for a given key
   
   (binding [atom/*current* 1000]
     (-> (cache {:type :mock
                 :initial {:a {:value 1 :expiration 1001}}})
         (cache/touch :a 10)
         :state
         deref))
   => {:a {:value 1, :expiration 11000}}"
  {:added "3.0"}
  [cache key expiry]
  (protocol.cache/-touch cache key expiry))

(defn expired?
  "checks if a given key time is expired
   
   (binding [atom/*current* 1000]
     (-> (cache {:type :mock
                 :initial {:a {:value 1 :expiration 1001}}})
         (cache/expired? :a)))
   => false"
  {:added "3.0"}
  [cache key]
  (protocol.cache/-expired? cache key))

(defn expiry
  "return the expiry of a key in seconds
 
   (binding [atom/*current* 1000]
     (-> (cache {:type :mock
                 :initial {:a {:value 1 :expiration 7000}}})
         (cache/expiry :a)))
   => 6"
  {:added "3.0"}
  [cache key]
  (protocol.cache/-expiry cache key))

(common/extend-impl [protocol.cache/ICacheExtend {:except #{:-mget :-mset :-bulk :-transact}}]
                    [protocol.cache/ICacheList {:except #{:-lpush :-rpush}}]
                    [protocol.cache/ICacheHash {:except #{:-hdel :-hmget :-hset}}]
                    [protocol.cache/ICacheSortedSet {:except #{:-zadd :-zrem}}])

(defn mget
  ([] [])
  ([cache k & more]
   (protocol.cache/-mget cache (cons k more))))

(defn mset
  ([] [])
  ([cache arg & more]
   (protocol.cache/-mset cache (cons arg more))))

(defmacro bulk
  "allows transaction operations"
  {:added "3.0"}
  [cache & body]
  `(protocol.cache/-bulk ~cache
                         (fn []
                           (vector ~@body))))

(defmacro transact
  [cache & body]
  `(protocol.cache/-transact ~cache
                             (fn []
                               (vector ~@body))))

(defn lpush
  "pushes elements at the head of the list
 
   (-> (doto (cache {:type :mock})
         (cache/delete :a)
         (cache/lpush :a 1 2 3 4 5 ))
       (cache/lrange :a 0 -1))
   => [5 4 3 2 1]"
  {:added "3.0"}
  [cache key element & more]
  (protocol.cache/-lpush cache key (cons element more)))

(defn rpush
  "pushes elements at the end of the list
 
   (-> (doto (cache {:type :mock})
         (cache/delete :a)
         (cache/rpush :a 1 2 3 4 5 ))
       (cache/lrange :a 0 -1))
   => [1 2 3 4 5]"
  {:added "3.0"}
  [cache key element & more]
  (protocol.cache/-rpush cache key (cons element more)))

(defn hdel
  "deletes keys in the hash
  
   (-> (doto (cache {:type :mock})
         (cache/delete :a)
         (cache/hset :a {\"a\" 1 \"b\" 2 \"c\" 3})
         (cache/hdel :a \"a\" \"c\"))
       (cache/hgetall :a))
   => [\"b\" 2]
 
   (-> (doto (cache {:type :redis})
         (cache/delete :a)
         (cache/hset :a {\"a\" 1 \"b\" 2 \"c\" 3})
         (cache/hdel :a \"a\" \"c\"))
       (cache/hgetall :a))
   => [\"b\" 2]"
  {:added "3.0"}
  ([cache key] [])
  ([cache key field & more]
   (protocol.cache/-hdel cache key (cons field more))))

(defn hmget
  "function for getting multiple values
   
   (-> (doto (cache {:type :mock})
         (cache/delete :a)
         (cache/hset :a {\"a\" 1 \"b\" 2 \"c\" 3}))
       (cache/hmget :a \"a\" \"b\"))
   => [1 2]
 
   
   (-> (doto (cache {:type :redis})
         (cache/delete :a)
         (cache/hset :a {\"a\" 1 \"b\" 2 \"c\" 3}))
       (cache/hmget :a \"a\" \"b\"))
   => [1 2]"
  {:added "3.0"}
  ([cache key] [])
  ([cache key field & more]
   (protocol.cache/-hmget cache key (cons field more))))

(defn hset
  "function for setting hashes"
  {:added "3.0"}
  ([cache key m]
   (protocol.cache/-hset cache key [m]))
  ([cache key field value & args]
   (protocol.cache/-hset cache key (concat [field value] args))))

(defn zadd
  "function for adding to sorted set"
  {:added "3.0"}
  ([cache key m]
   (protocol.cache/-zadd cache key [m]))
  ([cache key field value & args]
   (protocol.cache/-zadd cache key (concat [field value] args))))

(defn zrem
  "function for removing form sorted set"
  {:added "3.0"}
  ([cache key] [])
  ([cache key member & more]
   (protocol.cache/-zrem cache key (cons member more))))

(defn create
  "creates a cache that is component compatible
 
   (cache/create {:type :mock
                  :file {:path \"dev/scratch/test.edn\"
                         :reset true}})
   ;;=> #cache.mock<uninitiased>
   "
  {:added "3.0"}
  [m]
  (protocol.cache/-create m))

(defn cache
  "creates an active cache
 
   (cache {:type :mock
           :initial {:a {:value 1}}
           :file {:path \"dev/scratch/test.edn\"
                  ;;:load true
                  :reset true}})
   ;;=> #cache.mock{:a 1}
   "
  {:added "3.0"}
  ([] (cache {:type :mock}))
  ([m]
   (-> (create m)
       (component/start))))


