(ns hara.platform.bank
  (:require [hara.protocol.bank :as protocol.bank]
            [hara.platform.bank.mock :as mock]
            [hara.core.component :as component]))

(defn get-account
  "returns account info
 
   (-> (create {:type :mock :accounts [{:id \"1\" :balances {\"USD\" 100}}]})
       (get-account \"1\"))
   => {:id \"1\", :name \"\", :balances {\"USD\" 100}}"
  {:added "1.0"}
  ([bank {:keys [id] :as data}]
   (protocol.bank/-get-account bank data)))

(defn add-account
  "adds an account to the bank
 
   (-> (create {:type :mock})
       (add-account {:id \"1\" :balances {\"USD\" 100}}))
   => (contains {:op :add-account, :id \"1\", :name \"\", :balances {\"USD\" 100}})"
  {:added "1.0"}
  ([bank {:keys [id name balances tx time] :as data}]
   (protocol.bank/-add-account bank data)))

(defn remove-account
  "deletes an account from the bank
 
   (-> (create {:type :mock :accounts [{:id \"1\" :balances {\"USD\" 100}}]})
       (remove-account {:id \"1\"}))
   => (contains {:op :remove-account, :id \"1\", :name \"\", :balances {\"USD\" 100}})"
  {:added "1.0"}
  ([bank {:keys [id tx time] :as data}]
   (protocol.bank/-remove-account bank data)))

(defn list-accounts
  "lists all accounts in the bank
 
   (-> (create {:type :mock :accounts [{:id \"1\" :balances {\"USD\" 100}}]})
       (list-accounts))
   => [\"1\"]"
  {:added "1.0"}
  ([bank]
   (list-accounts bank {}))
  ([bank opts]
   (protocol.bank/-list-accounts bank opts)))

(defn deposit
  "deposits amount into an account
 
   (-> (create {:type :mock :accounts [{:id \"1\" :balances {\"USD\" 100}}]})
       (deposit {:id \"1\" :currency \"USD\" :amount 200}))
   
   => (contains {:op :adjust-account, :id \"1\", :currency \"USD\" :amount 200, :old 100, :new 300})"
  {:added "1.0"}
  ([bank {:keys [id currency amount tx time] :as data}]
   (protocol.bank/-deposit bank data)))

(defn withdraw
  "withdraws amount from an account
 
   (-> (create {:type :mock :accounts [{:id \"1\" :balances {\"USD\" 1000}}]})
       (withdraw {:id \"1\" :currency \"USD\" :amount 200}))
   => (contains {:op :adjust-account, :id \"1\", :currency \"USD\" :amount -200, :old 1000, :new 800})"
  {:added "1.0"}
  ([bank {:keys [id currency amount tx time] :as data}]
   (protocol.bank/-withdraw bank data)))

(defn bank?
  "checks if an object is of type bank
 
   (bank? (create {:type :mock}))
   => true"
  {:added "3.0"}
  ([obj]
   (satisfies? protocol.bank/IBank obj)))

(defn create
  "creates a bank
 
   (create {:type :mock :name \"UNION INC.\"})
   ;; #bank{:name \"UNION INC.\", :accounts []}
   => hara.platform.bank.mock.MockBank"
  {:added "1.0"}
  ([m]
   (protocol.bank/-create m)))

(defn bank
  "constructs a componentizable bank
 
   (bank {:type :mock :name \"UNION INC.\"})
   ;; #bank{:name \"UNION INC.\", :accounts []}
   => bank?"
  {:added "3.0"}
  ([m]
   (-> (create m)
       (component/start))))


