(ns hara.platform.exchange
  (:require [hara.protocol.exchange :as protocol.exchange]
            [hara.core.component :as component]
            [hara.platform.exchange.base :deps true])
  (:refer-clojure :exclude [queue]))

(defn list-queues
  "returns current list of queues
 
   (list-queues (exchange {:type :mock
                           :routing routes}))
   => (contains {\"q1\" map?
                 \"q2\" map?})"
  {:added "3.0"}
  ([mq]
   (protocol.exchange/-list-queues mq)))

(defn add-queue
  "adds a queue to the mq
 
   (-> (exchange {:type :mock
                  :routing routes})
       (add-queue \"q3\")
       (list-queues))
   => (contains {\"q1\" map?
                 \"q2\" map?
                 \"q3\" map?})"
  {:added "3.0"}
  ([mq name]
   (add-queue mq name {}))
  ([mq name opts]
   (protocol.exchange/-add-queue mq name opts)))

(defn delete-queue
  "deletes a queue from the mq
 
   (-> (exchange {:type :mock
                  :routing routes})
       (delete-queue \"q1\")
       (list-queues))
   => (contains {\"q2\" map?})"
  {:added "3.0"}
  ([mq name]
   (protocol.exchange/-delete-queue mq name)))

(defn list-exchanges
  "returns current list of exchanges
 
   (list-exchanges (exchange {:type :mock
                              :routing routes}))
   => (contains {\"ex1\" map?
                 \"ex2\" map?})"
  {:added "3.0"}
  ([mq]
   (protocol.exchange/-list-exchanges mq)))

(defn add-exchange
  "adds an exchange to the mq
 
   (-> (exchange {:type :mock
                  :routing routes})
       (add-exchange \"ex3\")
       (list-exchanges))
   => (contains {\"ex1\" map?
                 \"ex2\" map?
                 \"ex3\" map?})"
  {:added "3.0"}
  ([mq name]
   (add-exchange mq name {}))
  ([mq name opts]
   (protocol.exchange/-add-exchange mq name opts)))

(defn delete-exchange
  "removes an exchange from the mq
 
   (-> (exchange {:type :mock
                  :routing routes})
       (delete-exchange \"ex1\")
       (list-exchanges))
   => (contains {\"ex2\" map?})"
  {:added "3.0"}
  ([mq name]
   (protocol.exchange/-delete-exchange mq name)))

(defn list-bindings
  "returns current list of exchanges
 
   (-> (exchange {:type :mock
                  :routing routes})
       (list-bindings))
   => (contains-in {\"ex1\" {:exchanges {\"ex2\" [map?]}
                           :queues {\"q1\" [map?]}}
                    \"ex2\" {:queues {\"q2\" [map?]}}})"
  {:added "3.0"}
  ([mq]
   (protocol.exchange/-list-bindings mq)))

(defn bind-exchange
  "binds a queue to the exchange
 
   (-> (exchange {:type :mock
                  :routing routes})
       (add-exchange \"ex3\")
       (bind-exchange \"ex1\" \"ex3\")
       (list-bindings))
   => (contains-in {\"ex1\" {:exchanges {\"ex2\" [map?]
                                       \"ex3\" [map?]}
                           :queues {\"q1\" [map?]}}
                   \"ex2\" {:queues {\"q2\" [map?]}}})"
  {:added "3.0"}
  ([mq source dest]
   (bind-exchange mq source dest {}))
  ([mq source dest opts]
   (protocol.exchange/-bind-exchange mq source dest opts)))

(defn bind-queue
  "binds an exchange to the exchange
 
   (-> (exchange {:type :mock :routing routes})
       (add-queue \"q3\")
       (bind-queue \"ex1\" \"q3\")
       (list-bindings))
   => (contains-in {\"ex1\" {:exchanges {\"ex2\" [map?]}
                           :queues {\"q1\" [map?]
                                    \"q3\" [map?]}}
                    \"ex2\" {:queues {\"q2\" [map?]}}})"
  {:added "3.0"}
  ([mq source dest]
   (bind-queue mq source dest {}))
  ([mq source dest opts]
   (protocol.exchange/-bind-queue mq source dest opts)))

(defn list-consumers
  "lists all the consumers for the mq
 
   (-> (exchange {:type :mock
                  :routing routes
                  :consumers consumers})
       (list-consumers))
   => (contains-in {\"q1\" {:hello map?,
                          :world map?},
                    \"q2\" {:foo map?}})"
  {:added "3.0"}
  ([mq]
   (protocol.exchange/-list-consumers mq)))

(defn add-consumer
  "adds a consumers to the m
 
   (-> (exchange {:type :mock
                  :routing routes
                  :consumers consumers})
       (add-consumer \"q2\" {:id :bar
                           :sync true
                           :function prn})
       (list-consumers))
   => (contains-in {\"q1\" {:hello map?
                         :world map?}
                    \"q2\" {:foo map?
                          :bar map?}})"
  {:added "3.0"}
  ([mq name handler]
   (protocol.exchange/-add-consumer mq name handler)))

(defn delete-consumer
  "deletes the consumer from the queue
   
   (-> (exchange {:type :mock
                  :routing routes
                  :consumers consumers})
       (delete-consumer \"q1\" :hello)
       (list-consumers))
   => (contains-in {\"q1\" {:world map?}
                    \"q2\" {:foo map?}})"
  {:added "3.0"}
  ([mq name id]
   (protocol.exchange/-delete-consumer mq name id)))

(defn publish
  "publishes a message to an exchange
 
   (def p (promise))
   
   (-> (exchange {:type :mock
                  :routing routes
                  :consumers {\"q1\" {:hello {:function #(deliver p %)}}}})
       (publish \"ex1\" \"hello there\"))
   
   @p => \"hello there\""
  {:added "3.0"}
  ([mq exchange message]
   (publish mq exchange message {}))
  ([mq exchange message opts]
   (protocol.exchange/-publish mq exchange message opts)))

(defn create
  "creates a exchange that is component compatible
 
   (exchange/create {:type :mock
                     :file {:path \"dev/scratch/test.edn\"
                            :reset true}})
   ;;=> #exchange.mock<uninitiased>
   "
  {:added "3.0"}
  [m]
  (protocol.exchange/-create m))

(defn exchange
  "creates an active exchange
 
   (exchange {:type :mock
              :routing routes
              :file {:path \"dev/scratch/test.edn\"
                     :reset true
                     :cleanup true}})
   ;;=> #exchange.mock
   ;;   {:queues #{\"q1\" \"q2\"},
   ;;    :exchanges #{\"ex1\" \"ex2\"},
   ;;    :bindings {\"ex1\" {:exchanges #{\"ex2\"},
   ;;                      :queues #{\"q1\"}},
   ;;               \"ex2\" {:queues #{\"q2\"},
   ;;                      :exchanges #{}}}}
   "
  {:added "3.0"}
  ([]
   (exchange {:type :mock}))
  ([m]
   (-> (create m)
       (component/start))))
