(ns hara.log.form
  (:require [hara.data.base.map :as map]
            [hara.core.base.util :as util]
            [hara.string :as string]
            [clojure.main :as main]
            [hara.log.common :as common]
            [hara.log.core :as core]))

(defn log-meta
  "gets the metadata information on macroexpand"
  {:added "3.0"}
  ([form]
   (let [{:keys [line column]} (meta form)
         ns    (str (.getName *ns*))]
     {:log/namespace ns
      :log/line      line
      :log/column    column})))

(defn log-function-name
  "gets the function name
 
   (log-function-name \"ns$function_name\")
   => \"function-name\""
  {:added "3.0"}
  [name]
  (-> (string/split name #"\$")
      (second)
      (main/demunge)))

(defn log-runtime-raw
  "returns the runtime information of the log entry
 
   (set (keys (log-runtime-raw nil \"hara.log.form-test\")))
   => #{:log/function :log/method :log/filename}"
  {:added "3.0"}
  [id ns]
  (let [e        (RuntimeException. "")
        frames   (.getStackTrace e)]
    (comment {:log/return (.printStackTrace e)})
    (->> frames
         (keep (fn [^StackTraceElement frame]
                 (let [name (.getClassName frame)]
                   (if (and (not (.startsWith name "hara.log.form$log_runtime"))
                            (.startsWith name (munge ns))
                            (.contains name "$"))
                     {:log/function (log-function-name name)
                      :log/method   (.getMethodName frame)
                      :log/filename (.getFileName frame)}))))
         first)))

(def log-runtime (memoize log-runtime-raw))

(defn log-check
  "checks to see if form should be ignored
 
   (log-check :debug)
   => true"
  {:added "3.0"}
  [level]
  (if (or (not common/*static*)
          (>= (get common/+levels+ level)
              (get common/+levels+ common/*level*)))
    true false))

(defn log-form
  "function for not including log forms on static compilation"
  {:added "3.0"}
  ([type form logger message data]
   (if (log-check type)
     (let [id   (util/sid)
           meta (log-meta form)]
       `(core/log-write ~logger ~type
                        (merge ~meta (log-runtime ~id ~(:log/namespace meta)))
                        ~message ~data)))))

(defmacro log
  "produces a log"
  {:added "3.0"}
  ([data]
   (log-form :info &form `common/*logger* "" data))
  ([message data]
   (log-form :info &form `common/*logger*  message data))
  ([type message data]
   (log-form type &form `common/*logger* message data))
  ([logger type message data]
   (log-form type &form logger message data)))

(defn log-macro-form
  "creates a standard form given a type"
  {:added "3.0"}
  [type]
  (list 'defmacro (symbol (name type))
        (list '[data]
              (list 'log-form type '&form (list 'quote 'hara.log.common/*logger*) "" 'data))
        (list '[message data]
              (list 'log-form type '&form (list 'quote 'hara.log.common/*logger*) 'message 'data))
        (list '[logger message data]
              (list 'log-form type '&form 'logger 'message 'data))))

(defmacro log-macros
  "creates forms for `debug`, `status`, `info`, `error`, `warn`, `fatal`"
  {:added "3.0"}
  [types]
  (mapv log-macro-form types))

(log-macros [:code :note :debug :status :info :todo :warn :thrown :error :fatal])

(defn profile-form
  "helper function for the `trace` macro"
  {:added "3.0"}
  [tag form logger message data body return]
  (cond (not (log-check tag))
        body

        :else
        (let [id   (util/sid)
              meta (log-meta form)
              form (pr-str body)]
          `(let [t#  (System/currentTimeMillis)
                 t0# (System/nanoTime)]
             (try
               (let [out# ~body]
                 (core/log-write ~logger ~tag
                                 (merge ~meta
                                        (log-runtime ~id ~(:log/namespace meta))
                                        (cond-> {:log/message ~message
                                                 :log/form ~form
                                                 :log/duration (- (System/nanoTime) t0#)
                                                 :log/outcome :ok}
                                          ~return (assoc :log/return out#)))
                                 ~message
                                 ~data
                                 t#)
                 out#)
               (catch Exception e#
                 (core/log-write ~logger ~tag
                                 (merge ~meta
                                        (log-runtime ~id ~(:log/namespace meta))
                                        (cond-> {:log/message ~message
                                                 :log/form ~form
                                                 :log/duration (- (System/nanoTime) t0#)
                                                 :log/outcome :error
                                                 :log/exception e#}))
                                 ~message
                                 ~data
                                 t#)
                 (throw e#)))))))

(defn profile-macro-form
  "creates a standard form given a type"
  {:added "3.0"}
  [type return]
  (list 'defmacro (symbol (name type))
        (list '[body]
              (list 'profile-form type '&form (list 'quote 'hara.log.common/*logger*) "" nil 'body return))
        (list '[opts body]
              (list 'if '(string? opts)
                    (list 'profile-form type '&form (list 'quote 'hara.log.common/*logger*) 'opts nil 'body return)
                    (list 'profile-form type '&form (list 'quote 'hara.log.common/*logger*) "" 'opts 'body return)))
        (list '[message data body]
              (list 'profile-form type '&form (list 'quote 'hara.log.common/*logger*) 'message 'data 'body return))
        (list '[logger message data body]
              (list 'profile-form type '&form 'logger 'message 'data 'body return))))

(defmacro profile-macros
  "creates forms for profiling macros"
  {:added "3.0"}
  [& types]
  (mapv #(apply profile-macro-form %) types))

(profile-macros [:spy true] [:silent true] [:trace false] [:meter false])
