(ns hara.log.console
  (:require [hara.protocol.log :as protocol.log]
            [hara.protocol.component :as protocol.component]
            [hara.io.concurrent :as cc]
            [hara.string.base.ansi :as ansi]
            [hara.log.core :as core]
            [hara.print.pretty :as pretty]
            [hara.string :as string]))

(defonce +global-queue+    (cc/queue))
(defonce +global-executor+ (cc/single-executor 1))

(defn logger-process-basic
  "process the basic logger"
  {:added "3.0"}
  ([{:keys [pretty]} items]
   (doseq [item items]
     (if (false? pretty)
       (printf "%s" (prn-str item))
       (println (pretty/pprint-str item) "\n")))
   (flush)))

(defrecord BasicLogger [instance]
  
  Object
  (toString [logger] (str "#log.basic" (core/logger-info-core logger)))

  protocol.log/ILogger
  (-logger-write [logger entry] (core/logger-write-core logger entry))
  
  protocol.log/ILoggerProcess
  (-logger-process [logger items] (logger-process-basic logger items)))

(defmethod print-method BasicLogger
  [v ^java.io.Writer w]
  (.write w (str v)))

(defmethod protocol.log/-create :basic
  [m]
  (-> (merge {:interval 200
              :max-batch 100
              :queue +global-queue+
              :executor +global-executor+}
             m)
      (core/logger-init-core)   
      (map->BasicLogger)))

(defn basic-logger
  "creates a basic logger
 
   (basic-logger {:interval 500 :max-batch 10000})"
  {:added "3.0"}
  ([] (basic-logger nil))
  ([m]
   (-> (protocol.log/-create (assoc m :type :basic))
       (protocol.component/-start))))

(defn duration-pretty
  [time digits]
  (let [full   (str time)
        len    (count full)
        suffix (cond (<= 1 len 3)
                     "ns"
                     (<= 4 len 6)
                     "us"
                     (<= 7 len 9)
                     "ms"
                     (<= 10 len 12)
                     "s")
        decimal (inc (rem (dec len) 3))]
    (if (<= len digits)
      (str full "ns")
      (let [out (subs full 0 digits)]
        (str (subs out 0 decimal)
             "."
             (subs out decimal)
             suffix)))))

(def +console-colors+
  {:silent  {:highlight false :color :white}
   :note    {:highlight true  :color :white}
   :code    {:highlight true  :color :white}
   :trace   {:highlight true  :color :cyan}
   :spy     {:highlight true  :color :cyan}
   :debug   {:highlight true  :color :cyan}
   :meter   {:highlight true  :color :green}
   :status  {:highlight true  :color :green}
   :info    {:highlight true  :color :green} 
   :todo    {:highlight true  :color :yellow}
   :warn    {:highlight true  :color :yellow}
   :thrown  {:highlight true  :color :blue}
   :error   {:highlight true  :color :red}
   :fatal   {:highlight true  :color :magenta}})

(defn console-label [level]
  (let [{:keys [highlight color]} (get +console-colors+ level)
        text (.toUpperCase (name level))
        text (if highlight
               (str " " text " ")
               (str " " text))
        len  (count text)
        out  (if (not highlight)
               (ansi/style text [:bold color])
               (ansi/style text [:bold :white (keyword (str "on-" (name color)))]))]
    [out len]))

(defn logger-process-console
  "processes the console logger"
  {:added "3.0"}
  ([_ items]
   (doseq [{:log/keys [function namespace line column level timestamp message value] :as entry} items]
     (let [label (if function
                   (str namespace "/" function)
                   namespace)
           [lstr len] (console-label level)
           entry (cond-> (dissoc entry
                                 :log/function :log/namespace :log/line :log/method :log/filename
                                 :log/column :log/level :log/timestamp :log/message :log/value)
                   (:log/duration entry) (update :log/duration duration-pretty 5))]
       (print 
        (str  lstr (apply str (repeat (- 9 len) " "))
              (ansi/white "["
                          (ansi/cyan (format "%s @ %d,%d" label line column))
                          "]"
                          (format " :: %s" (java.util.Date. ^long timestamp)))
              "\n"
              (if-not (empty? message) (ansi/green message  "  "))
              (let [out (if (and value (empty? entry))
                          (pretty/pprint-str value)
                          (pretty/pprint-str entry))]
                (if (pos? (.indexOf ^String out "\n"))
                  (str "\n" out)
                  out))
              "\n\n"))))
   (flush)))


(defrecord ConsoleLogger [instance]
  
  Object
  (toString [logger] (str "#log.console" (core/logger-info-core logger)))

  protocol.log/ILogger
  (-logger-write [logger entry] (core/logger-write-core logger entry))
  
  protocol.log/ILoggerProcess
  (-logger-process [logger items]
    (logger-process-console logger items)))

(defmethod print-method ConsoleLogger
  [v ^java.io.Writer w]
  (.write w (str v)))

(defmethod protocol.log/-create :console
  [m]
  (-> (merge {:interval 200
              :max-batch 100
              :queue +global-queue+
              :executor +global-executor+}
             m)
      (core/logger-init-core)
      (map->ConsoleLogger)))

(defn console-logger
  "creates a console logger
 
   (console-logger  {:interval 500 :max-batch 10000})"
  {:added "3.0"}
  ([] (console-logger nil))
  ([m]
   (-> (protocol.log/-create (assoc m :type :console))
       (protocol.component/-start))))

