(ns hara.log.analyser
  (:require [hara.io.file :as fs]
            [hara.core.base.util :as util]
            [hara.data.base.map :as map]))
            
(require '[hara.code.framework :as framework])

(def ^:dynamic *toplevel-forms*
  '#{def-
     def
     defn
     defn-
     defmulti
     defmethod
     defmacro})

(defn ns-file
  "returns the file
 
   (ns-file 'hara.test)
   => \"hara/test.clj\""
  {:added "3.0"}
  [^clojure.lang.Symbol ns]
  (-> (str ns)
      (.replaceAll "\\." "/")
      (.replaceAll "-" "_")
      (str ".clj")))

(defn ns-block-analyse-input
  "returns the input for an analysis
 
   (ns-block-analyse-input 'hara.test)
   ;; [\"file:/Users/chris/Development/caudata/hara/src/hara/test.clj\" 1574706839459]
   => (contains [string? number?])"
  {:added "3.0"}
  [ns]
  (let [url (fs/resource (ns-file ns))]
    (cond (= "file" (.getProtocol url))
          (let [path (fs/file (.getPath url))]
            [(str url) (.lastModified path)])

          :else
          [(str url) 0])))

(defn ns-block-analyse-raw
  "returns ranges for functions, helper for `ns-block`
   
   (ns-block-analyse-raw (ns-block-analyse-input 'hara.test) 'hara.test)"
  {:added "3.0"}
  [[path t] ns]
  (binding [framework/*toplevel-forms* *toplevel-forms*]
    (->> (get (framework/analyse-source-code (slurp (util/url path)))
              ns)
         (map/map-vals (comp :line :source))
         (mapv (fn [[sym {:keys [row end-row]}]]
                 [row end-row sym])))))

(def ns-block-analyse (memoize ns-block-analyse-raw))

(defn ns-block
  "returns the ranges
 
   (ns-block 'hara.test)
   => '[[31 43 display-errors]
        [45 48 retrieve-fn]
        [50 53 test-lookup]
        [55 86 task/task-defaults]
        [101 128 print-options]
        [130 145 process-args]
        [147 158 -main]
       [160 170 run-errored]]"
  {:added "3.0"}
  [^clojure.lang.Symbol ns]
  (let [input (ns-block-analyse-input ns)]
    (ns-block-analyse input ns)))

(defn ns-function
  "finds the function given a namespace and line
 
   (ns-function 'hara.test 36)
   => \"display-errors\""
  {:added "3.0"}
  [ns line]
  (first (keep (fn [[start end sym]]
                 (if (<= start line end)
                   (str sym)))
               (ns-block ns))))

(defn ns-meta
  "returns the ns meta"
  {:added "3.0"}
  ([form]
   (let [{:keys [line column]} (meta form)
         ns    (str (.getName *ns*))
         func  (ns-function (.getName *ns*) line)
         timestamp (System/currentTimeMillis)]
     {:log/function  func
      :log/namespace ns
      :log/line      line
      :log/column    column})))

