(ns hara.log
  (:require [hara.protocol.log :as protocol.log]
            [hara.log.console :as console]
            [hara.log.core :as core]
            [hara.log.common :as common]
            [hara.log.form :as form]
            [hara.core.component :as component]
            [hara.module :as module]))

(module/include
 (hara.log.common put-logger!
                  set-logger!
                  set-static!
                  set-level!
                  set-context!)
 
 (hara.log.form log
                silent
                code
                note
                trace
                spy
                debug
                meter
                status
                info
                todo
                warn
                thrown
                error
                fatal))

(defonce +init+ (do (common/set-logger! (console/console-logger {:level :status}))
                    true))

(defn create
  "creates a component compatible logger
 
   (log/create {:type :basic})"
  {:added "3.0"}
  [m]
  (protocol.log/-create m))

(defn logger
  "creates an starts a logger
 
   (log/logger {:type :basic})"
  {:added "3.0"}
  [m]
  (-> (create m)
      (component/start)))

(defn logger?
  "checks if the object is a logger
 
   (log/logger? (log/create {:type :basic}))
   => true"
  {:added "3.0"}
  [m]
  (satisfies? protocol.log/ILogger m))

(defmacro with-logger
  "executes code with a logger"
  {:added "3.0"}
  [logger & body]
  `(binding [common/*logger* ~logger]
     ~@body))

(defmacro with-context
  "executes code with a logger"
  {:added "3.0"}
  [context & body]
  `(binding [common/*context* (merge common/*context* ~context)]
     ~@body))

(defmacro with
  "enables targeted printing of statements"
  {:added "3.0" :style/indent 1}
  [opts & body]
  `(let [~'{:keys [instance] :as logger} (or common/*logger* (console/console-logger))
         ~'logger (assoc ~'logger :instance (atom (merge (deref ~'instance) ~opts)))]
     (binding [common/*logger* ~'logger]
       ~@body)))

(defmacro with-debug
  "sets the current logger in debug mode"
  {:added "3.0"}
  [opts & body]
  (let [[opts body] (if (map? opts)
                      [(assoc opts :level :debug) body]
                      [{:level :debug} (cons opts body)])]
    `(with ~opts ~@body)))

(comment
  (with-logger (console/basic-logger)
    (todo "hello" {:rx 1}))
  (with-debug {:namespace {:include [#"log"]}}
    (todo "SERVER STARTED" {:rx 1}))
  (with-debug {:namespace {:include [#"log"]}}
    (trace {:aoeuoaeu 1 "aeouaoeuao" 1} (+ 3 4))))
