(ns hara.lib.undertow
  (:require [hara.io.file :as fs]
            [hara.string :as string])
  (:import (java.nio ByteBuffer)
           (java.io File InputStream FileInputStream)
           (java.util Map)
           (io.undertow Handlers Undertow Undertow$Builder)
           (io.undertow.io Sender)
           (io.undertow.server HttpHandler HttpServerExchange)
           (io.undertow.util HeaderMap HttpString HeaderValues Headers)
           (io.undertow.server.handlers BlockingHandler)))

(defn get-headers
  "gets headers from a io.undertow.util.HeaderMap"
  {:added "3.0"}
  [^HeaderMap header-map]
  (persistent!
   (reduce
    (fn [headers ^Iterable entry]
      (let [k (-> entry (. getHeaderName) str .toLowerCase keyword)
            val (if (> (.size entry) 1)
                  (set (iterator-seq (.iterator entry)))
                  (.get entry 0))]
        (assoc! headers k val)))
    (transient {})
    header-map)))

(defn set-headers
  "sets headers to a io.undertow.util.HeaderMap"
  {:added "3.0"}
  [^HeaderMap header-map headers]
  (reduce-kv
   (fn [^HeaderMap header-map ^String key val-or-vals]
     (let [key (HttpString. (name key))]
       (if (string? val-or-vals)
         (.put header-map key ^String val-or-vals)
         (.putAll header-map key val-or-vals)))
     header-map)
   header-map
   headers))

(defn keyword-path
  "creates a keyword path for a parameter
 
   (keyword-path \"a[b][c]\")
   => [:a :b :c]"
  {:added "3.0"}
  [key]
  (let [[_ k ks] (re-matches #"(?s)(.*?)((?:\[.*?\])*)" key)
        keys     (if ks (map second (re-seq #"\[(.*?)\]" ks)))]
    (map keyword (cons k keys))))

(defn get-params
  "gets params from .getQueryParameters"
  {:added "3.0"}
  [m]
  (reduce (fn [out [k v]]
            (let [v (if (> (.size v) 1)
                      (set (iterator-seq (.iterator v)))
                      (.getFirst v))]
              (assoc-in out (keyword-path k) v)))
          {}
          m))

(defn exchange-map
  "creates a map from a HttpServerExchange object"
  {:added "3.0"}
  [^HttpServerExchange exchange]
  (let [headers (.getRequestHeaders exchange)
        ctype   (.getFirst headers Headers/CONTENT_TYPE)]
    {:scheme   (-> exchange .getRequestScheme .toString .toLowerCase keyword)
     :method   (-> exchange .getRequestMethod .toString .toLowerCase keyword)
     :headers  (get-headers headers)
     :route    (-> exchange .getRequestPath)
     :body     (.getInputStream exchange)
     :server   {:name      (-> exchange .getHostName)
                :port      (-> exchange .getDestinationAddress .getPort)}
     :remote   (let [remote (-> exchange .getSourceAddress)]
                 {:ip   (->  remote .getAddress .getHostAddress)
                  :port (.getPort remote)})
     :query    {:string     (-> exchange .getQueryString)
                :params     (-> exchange .getQueryParameters get-params)}
     :content  {:length     (-> exchange .getRequestContentLength)
                :encoding (or (when ctype (Headers/extractTokenFromHeader ctype "charset"))
                              "UTF-8")}}))

(defprotocol RespondBody
  (respond [_ ^HttpServerExchange exchange]))

(extend-protocol RespondBody
  String
  (respond [body ^HttpServerExchange exchange]
    (.send ^Sender (.getResponseSender exchange) body))

  InputStream
  (respond [body ^HttpServerExchange exchange]
    (with-open [^InputStream b body]
      (.transferTo body (.getOutputStream exchange))))

  File
  (respond [f exchange]
    (respond (fs/input-stream f) exchange))
  
  nil
  (respond [_ exc]))

(defn set-exchange-response
  "sets the response for the HttpServerExchange object"
  {:added "3.0"}
  [^HttpServerExchange exchange {:keys [status headers body]}]
  (when-not exchange
    (throw (Exception. "Null exchange given.")))
  (when status
    (.setResponseCode exchange status))
  (set-headers (.getResponseHeaders exchange) headers)
  (respond body exchange))

;;; Adapter stuff

(defn undertow-handler
  "creates an undertow-handler"
  {:added "3.0"}
  [handler non-blocking]
  (reify
    HttpHandler
    (handleRequest [_ exchange]
      (when-not non-blocking
        (.startBlocking exchange))
      (let [request-map  (exchange-map exchange)
            response-map (handler request-map)]
        (set-exchange-response exchange response-map)))))

(defn on-io-proxy
  "creates a non-blocking handler"
  {:added "3.0"}
  [handler]
  (undertow-handler handler false))

(defn dispatching-proxy
  "creates a blocking handler"
  {:added "3.0"}
  [handler]
  (BlockingHandler. (undertow-handler handler true)))

(defn ^Undertow run-server
  "Start an Undertow webserver to serve the given handler according to the
   supplied options:
 
   :configurator   - a function called with the Undertow Builder instance
   :port           - the port to listen on (defaults to 80)
   :host           - the hostname to listen on
   :io-threads     - number of threads to use for I/O (default: number of cores)
   :worker-threads - number of threads to use for processing (default: io-threads * 8)
   :dispatch?      - dispatch handlers off the I/O threads (default: true)
 
   Returns an Undertow server instance. To stop call (.stop server)."
  {:added "3.0"}
  [handler {:keys [host port dispatch?]
            :or   {host "localhost"
                   port 80
                   dispatch? true}
            :as   options}]
  (let [^Undertow$Builder b (Undertow/builder)
        handler-proxy (if dispatch? dispatching-proxy on-io-proxy)]
    (.addListener b port host)
    (.setHandler b (handler-proxy handler))

    (let [{:keys [io-threads worker-threads]} options]
      (when io-threads     (.setIoThreads b io-threads))
      (when worker-threads (.setWorkerThreads b worker-threads)))

    (when-let [configurator (:configurator options)]
      (configurator b))

    (let [^Undertow s (.build b)]
      (.start s)
      s)))

(defmethod print-method io.undertow.io.UndertowInputStream
  [v w]
  (.write w (str "#stream" {:available (.available v)})))
